/*
 * @(#)ScribbleTool.java 5.1
 *
 */
package CH.ifa.draw.figures;

import java.awt.Point;
import java.awt.event.MouseEvent;

import CH.ifa.draw.framework.DrawingEditor;
import CH.ifa.draw.standard.UndoableTool;

/**
 * Tool to scribble a PolyLineFigure.
 * <p>
 * {@link PolyLineFigure}
 */
public class ScribbleTool extends UndoableTool {
    private PolyLineFigure fScribble;

    /**
     * The latest location of the received point.
     */
    private Point lastPoint;

    /**
     * Flag used to track whether the user has performed a drag action with the scribble tool to draw a shape.
     * Its purpose is to differentiate between a simple click action, which adds a point to the shape, and a drag action.
     */
    private boolean didDrag = false;

    /**
     * The last point of the MouseDown event.
     */
    private Point lastMouseDown;

    /**
     * Indicates whether smooth curves should be used in the scribble tool.
     * By default, the value is set to false, meaning points are connected by straight lines.
     */
    public boolean shouldUseSmoothCurves = false;

    /**
     * Normalized smoothness value used when the 'shouldUseSmoothCurves' option is selected.
     * The value ranges from 0.0 to 1.0. By default, the value is set to 0.5.
     */
    public static double smoothness = 0.5;

    /**
     * Creates a new ScribbleTool for drawing polyline figures.
     *
     * @param editor the DrawingEditor context in which this tool operates
     */
    public ScribbleTool(DrawingEditor editor) {
        super(editor);
    }

    @Override
    public void activate() {
        super.activate();
        fScribble = null;
    }

    @Override
    public void deactivate() {
        if (fScribble != null) {
            if (fScribble.size().width < 4 || fScribble.size().height < 4) {
                drawing().remove(fScribble);
                noChangesMade();
            }
        }
        fScribble = null;
        super.deactivate();
    }

    private void point(int x, int y) {
        if (fScribble == null) {
            if (shouldUseSmoothCurves) {
                // A. Create a new instance of SmoothPolyLineFigure
                fScribble = new SmoothPolyLineFigure(x, y, ScribbleTool.smoothness);
            } else {
                // B. Create a new instance of PolyLineFigure.
                fScribble = new PolyLineFigure(x, y);
            }

            view().add(fScribble);
            changesMade();
        } else if (lastPoint.x != x || lastPoint.y != y) {
            fScribble.addPoint(x, y);
        }

        lastPoint = new Point(x, y);
    }

    @Override
    public void mouseDown(MouseEvent e, int x, int y) {
        if (e.getClickCount() >= 2) {
            editor().toolDone();
        } else {
            // use original event coordinates to avoid
            // supress that the scribble is constrained to
            // the grid
            // EDIT: For the zooming feature to work we DO need the
            // transformed coordinates
            point(x, y);
        }

        // Store the mouseDown point to check later if user has dragged with the scribble tool
        lastMouseDown = new Point(x, y);
    }

    @Override
    public void mouseDrag(MouseEvent e, int x, int y) {
        if (fScribble != null) {
            point(x, y);
        }

        // Update the status of didDrag
        Point currentPoint = new Point(x, y);
        if (lastMouseDown != null) {
            boolean farEnough = currentPoint.distance(lastMouseDown) > 2;
            if (farEnough) {
                didDrag = true;
            }
        }

    }

    @Override
    public void mouseUp(MouseEvent e, int x, int y) {
        super.mouseUp(e, x, y);

        // Reset the shape to enable the creation of a new shape after the user has dragged with the scribble tool
        if (didDrag) {
            fScribble = null;
        }
        didDrag = false;
    }

}