package CH.ifa.draw.io;

import java.awt.Component;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;

import CH.ifa.draw.DrawPlugin;
import CH.ifa.draw.application.DrawApplication;
import CH.ifa.draw.application.MenuManager;
import CH.ifa.draw.framework.Drawing;
import CH.ifa.draw.io.exportFormats.ExportFormat;
import CH.ifa.draw.io.exportFormats.ExportFormatCommand;
import CH.ifa.draw.io.exportFormats.ExportFormatMulti;
import CH.ifa.draw.standard.NullDrawing;
import CH.ifa.draw.util.Command;
import CH.ifa.draw.util.CommandMenu;
import de.renew.util.StringUtil;


/**
 * Maintains a list of all known export formats ({@link ExportFormat})
 * by implementing the {@link ExportHolder} interface.
 * Also generates the Export menu ({@link #getExportMenu()})
 * and offers convenience methods to export to a specific format.
 */
public class ExportHolderImpl implements ExportHolder {
    /**
     * Creates log4j Logger for this class to represent logging information.
     */
    public static org.apache.log4j.Logger logger =
        org.apache.log4j.Logger.getLogger(ExportHolderImpl.class);
    private CommandMenu exportMenu;

    // All ExportFormats that have been added.
    private List<ExportFormat> _exportFormats;

    // The ExportMenu
    private CommandMenu _exportMenu11;
    private CommandMenu _exportMenuNN;
    private CommandMenu _exportMenuN1;

    // Static export mode appendix
    private static final String APPENDIX_11 = " current drawing ...";
    private static final String APPENDIX_N1 = " all drawings (N to 1) ...";
    private static final String APPENDIX_NN = " all drawings (N to N)";

    /**
     * Constructs and initializes ExportHolder with a new command menu.
     */
    public ExportHolderImpl() {
        super();
        exportMenu = new CommandMenu("Export");
        exportMenu.putClientProperty(MenuManager.ID_PROPERTY, "ch.ifa.draw.io.export");
        initExportFormatHolder();
    }

    private StatusDisplayer displayer() {
        return DrawPlugin.getGui();
    }

    private DrawApplication application() {
        return DrawPlugin.getGui();
    }

    /**
     * Returns the export menu for the export holder.
     *
     * @return the export menu
     */
    public JMenu getExportMenu() {
        assert (exportMenu != null) : "Failure in GuiPlugin: exportMenu == null";
        return exportMenu;
    }

    //	---------------------------------------------------------------------
    // Implementation of the ExportHolder Interface
    // ---------------------------------------------------------------------

    /**
      * Initiation of the ExportHolder.
      */
    private void initExportFormatHolder() {
        setExportFormats(new ArrayList<ExportFormat>());
        setExportMenu11(new CommandMenu("Export current drawing"));
        setExportMenuNN(new CommandMenu("Export all drawings (single file each)"));
        setExportMenuN1(new CommandMenu("Export all drawings (merged file)"));
        buildExportAll();
        exportMenu.add(exportMenu11());
        exportMenu.add(exportMenuNN());
        exportMenu.add(exportMenuN1());
    }

    /**
      * List of all added ExportFormats.
      * @ensure _exportFormats != null
      * @return Map list of all sdded ExportFormats
      */
    private List<ExportFormat> exportFormats() {
        assert (_exportFormats != null) : "Failure in GuiPlugin: _exportFormats == null";
        return _exportFormats;
    }

    /**
      * Sets the List of the ExportFormats
      * @require exportFormat != null
      * @ensure exportFormats() != null
      * @ensure exportFormats().equals(exportFormats)
      * @param exportFormats the value to be set
      */
    private void setExportFormats(List<ExportFormat> exportFormats) {
        _exportFormats = exportFormats;
        assert (exportFormats() != null) : "Failure in GuiPlugin: exportFormats == null";
        assert (exportFormats().equals(exportFormats))
            : "Failure in GuiPlugin: exportFormats != exportFormats()";
    }

    /**
      * Returns the ExportMenu.
      * @ensure _exportMenu11 != null
      * @return CommandMenu the ExportMenu of the ExportHolder
      */
    private CommandMenu exportMenu11() {
        assert (_exportMenu11 != null) : "Failure in GuiPlugin: _exportMenu11 == null";
        return _exportMenu11;
    }

    /**
      * Sets the _exportMenu to exportMenu.
      * @require exportMenu != null
      * @ensure exportMenu != null
      * @ensure exportMenu.equals(exportMenu)
      * @param exportMenu11 the value to be set
           */
    private void setExportMenu11(CommandMenu exportMenu11) {
        _exportMenu11 = exportMenu11;
        assert (exportMenu11() != null) : "Failure in GuiPlugin: exportMenu == null";
        assert (exportMenu11().equals(exportMenu11))
            : "Failure in GuiPlugin: exportMenu != exportMenu";
    }

    /**
      * Returns the ExportMenu.
      * @ensure _exportMenuNN != null
      * @return CommandMenu the ExportMenu of the ExportHolder
      */
    private CommandMenu exportMenuNN() {
        assert (_exportMenuNN != null) : "Failure in GuiPlugin: _exportMenuNN == null";
        return _exportMenuNN;
    }

    /**
      * Sets the _exportMenu to exportMenu.
      * @require exportMenu != null
      * @ensure exportMenu != null
      * @ensure exportMenu.equals(exportMenu)
      * @param exportMenuNN the value to be set
      */
    private void setExportMenuNN(CommandMenu exportMenuNN) {
        _exportMenuNN = exportMenuNN;
        assert (exportMenuNN() != null) : "Failure in GuiPlugin: exportMenu == null";
        assert (exportMenuNN().equals(exportMenuNN))
            : "Failure in GuiPlugin: exportMenu != exportMenu";
    }

    /**
      * Returns the ExportMenu.
      * @ensure _exportMenuN1 != null
      * @return CommandMenu the ExportMenu of the ExportHolder
      */
    private CommandMenu exportMenuN1() {
        assert (_exportMenuN1 != null) : "Failure in GuiPlugin: _exportMenuN1 == null";
        return _exportMenuN1;
    }

    /**
      * Sets the _exportMenu to exportMenu.
      * @require exportMenu != null.
      * @ensure exportMenu != null.
      * @ensure exportMenu.equals(exportMenu)
      * @param exportMenuN1 the value to be set
      */
    private void setExportMenuN1(CommandMenu exportMenuN1) {
        _exportMenuN1 = exportMenuN1;
        assert (exportMenuN1() != null) : "Failure in GuiPlugin: exportMenu == null";
        assert (exportMenuN1().equals(exportMenuN1))
            : "Failure in GuiPlugin: exportMenu != exportMenu";
    }


    /**
     * Returns the list of all FileFilters (Does not contain CombinationFileFilters).
     * @ensure result != null.
     * @return List of all FileFilters.
     */
    private SimpleFileFilter[] fileFilterExport(Drawing drawing) {
        List<FileFilter[]> fileFilters = new ArrayList<FileFilter[]>();
        ExportFormat[] formats = allExportFormats();
        for (int pos = 0; pos < formats.length; pos++) {
            if (formats[pos].canExportDrawing(drawing)) {
                FileFilter[] filters = buildFileFilter(formats[pos]);
                fileFilters.add(filters);
            }
        }

        List<SimpleFileFilter> allFileFilters = new ArrayList<SimpleFileFilter>();
        allFileFilters.add(new NoFileFilter());
        Iterator<FileFilter[]> iter = fileFilters.iterator();
        while (iter.hasNext()) {
            FileFilter[] element = iter.next();
            for (int pos = 0; pos < element.length; pos++) {
                SimpleFileFilter current = (SimpleFileFilter) element[pos];

                // work around
                boolean exists = false;
                for (int pos2 = 0; pos2 <= pos; pos2++) {
                    if (current.equals(allFileFilters.get(pos2))) {
                        exists = true;
                        break;
                    }
                }
                if (!exists) {
                    allFileFilters.add(current);
                }
            }
        }

        return allFileFilters.toArray(new SimpleFileFilter[0]);
    }

    private FileFilter[] buildFileFilter(ExportFormat exportFormat) {
        FileFilter filter = exportFormat.fileFilter();
        List<FileFilter> list = new ArrayList<FileFilter>();
        if (filter instanceof CombinationFileFilter) {
            CombinationFileFilter comFilter = (CombinationFileFilter) filter;
            Iterator<SimpleFileFilter> filters = comFilter.getFileFilters().iterator();
            while (filters.hasNext()) {
                list.add(filters.next());
            }
        } else {
            list.add(filter);
        }

        return list.toArray(new FileFilter[0]);
    }

    /**
     * Constructs the menu item exportAll by using a Command.
     */
    private void buildExportAll() {
        Command command = new Command("Export current drawing (any type)...") {
            @Override
            public void execute() {
                if (!(application().drawing() instanceof NullDrawing)) {
                    File path = DrawPlugin.getCurrent().getIOHelper().getSaveFile(
                        null, fileFilterExport(application().drawing()), application().drawing());
                    if (path != null) {
                        List<ExportFormat[]> list = new ArrayList<ExportFormat[]>();
                        for (int pos = 0; pos < allExportFormats().length; pos++) {
                            ExportFormat[] formats = allExportFormats()[pos].canExport(path);
                            if (formats.length > 0) {
                                list.add(formats);
                            }
                        }
                        Iterator<ExportFormat[]> formatsIter = list.iterator();
                        List<ExportFormat> allFormats = new ArrayList<ExportFormat>();
                        while (formatsIter.hasNext()) {
                            ExportFormat[] formatArray = formatsIter.next();
                            for (int pos = 0; pos < formatArray.length; pos++) {
                                if (formatArray[pos].canExportDrawing(application().drawing())) {
                                    if (new NoFileFilter().equals(formatArray[pos].fileFilter())) {
                                        if (StringUtil.getExtension(path.getPath()).equals("")) {
                                            allFormats.add(formatArray[pos]);
                                        }
                                    } else {
                                        allFormats.add(formatArray[pos]);
                                    }
                                }
                            }
                        }
                        ExportFormat format = null;
                        if (allFormats.size() == 1) {
                            format = allFormats.get(0);
                        } else if (allFormats.size() > 1) {
                            Object choice = JOptionPane.showInputDialog(
                                null, "Choose", "ExportFormats", JOptionPane.OK_CANCEL_OPTION, null,
                                allFormats.toArray(), allFormats.get(0));
                            if (choice != null) {
                                format = (ExportFormat) choice;
                            }
                        }
                        if (format != null) {
                            saveDrawing(application().drawing(), format, path);
                        } else {
                            displayer().showStatus("no ExportFormat");
                        }
                    }
                } else {
                    displayer().showStatus("no drawing");
                }
            }

            @Override
            public boolean isExecutable() {
                return super.isExecutable() && application() != null
                    && !(application().drawing() instanceof NullDrawing);
            }
        };
        exportMenu.add(command);
    }

    /**
      * Save an enumeration of drawings with the help of format (n to 1).
      * @require format != null.
      * @require drawings != null
      * @require path != null
      * @param drawings the enumeration of drawings to be saved
      * @param format the format in which the enumeration is to be saved
      * @param path the filename of the file the enumeration is to be saved to
      * @param displ the StatusDisplayer the export message will be shown in
      */
    public void saveDrawings(
        Enumeration<Drawing> drawings, ExportFormat format, File path, StatusDisplayer displ)
    {
        try {
            ArrayList<Drawing> list2 = Collections.list(drawings);
            Drawing[] array = list2.toArray(new Drawing[0]);
            format.export(array, path);
            displ.showStatus("Exported " + path.getPath() + ".");
        } catch (Exception e) {
            logger.error(e.getMessage());
            displ.showStatus(e.toString());
            if (logger.isDebugEnabled()) {
                logger.debug(ExportHolderImpl.class.getSimpleName() + ": ", e);
            }
        }
    }

    /**
     * Save an enumeration of drawings with the help of format.
     * @require format != null.
     * @require drawings != null
     * @param drawings
     * @param format
     */
    private void saveDrawings(Enumeration<Drawing> drawings, ExportFormat format) {
        try {
            List<Drawing> drawingList = new ArrayList<Drawing>();
            while (drawings.hasMoreElements()) {
                Drawing drawing = drawings.nextElement();

                if (drawing.getFilename() == null) {
                    // save drawing         
                    application().saveDrawingAs(drawing);
                } else {
                    // add Drawing to list
                    drawingList.add(drawing);
                }
            }

            // list to array
            Drawing[] drawingArray = drawingList.toArray(new Drawing[0]);

            // generate paths
            File[] paths = new File[drawingArray.length];
            for (int pos = 0; pos < drawingArray.length; pos++) {
                String name = drawingArray[pos].getName();
                File path = drawingArray[pos].getFilename().getCanonicalFile();
                String pathString = path.getParent() + File.separator + name;
                paths[pos] = DrawingFileHelper.checkAndAddExtension(
                    new File(pathString), (SimpleFileFilter) format.fileFilter());
            }

            if (drawingArray.length > 0) {
                format.exportAll(drawingArray, paths);
                displayer().showStatus("Exported.");
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            displayer().showStatus(e.toString());
        }
    }

    /**
     * Save an array of drawings with the help of format.
     * @require drawing != null
     * @require format != null
     * @require path != null
     * @param drawing The drawing to be saved
     * @param format The format in which the drawing is to be saved
     * @param path The filename of the file the drawing is to be saved to
     * @param sd The StatusDisplayer the export message will be shown in
     */
    public void saveDrawing(Drawing drawing, ExportFormat format, File path, StatusDisplayer sd) {
        try {
            File pathResult = format.export(drawing, path);
            sd.showStatus("Exported " + pathResult.getPath() + ".");
        } catch (Exception e) {
            sd.showStatus(e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Save an array of drawings with the help of format.
     * @require drawing != null
     * @require format != null
     * @require path != null
     * @param drawing The drawing to be saved
     * @param format The format
     * @param path The file to be saved to
     */
    private void saveDrawing(Drawing drawing, ExportFormat format, File path) {
        try {
            File pathResult = format.export(drawing, path);
            displayer().showStatus("Exported " + pathResult.getPath() + ".");
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            displayer().showStatus(e.toString());
        }
    }


    /**
     * Constructs a menu item for this exportFormat in exportMenu.
     * @require exportFormat != null.
     * @require parent != null.
     * @param exportFormat the ExportFormat for the new menu item
     * @param parent11 The parent menu from 1 to 1
     * @param parentNN The parent menu from n to n
     * @param parentN1 The parent menu from n to 1
     */
    private void buildExportFormat(
        ExportFormat exportFormat, CommandMenu parent11, CommandMenu parentNN, CommandMenu parentN1)
    {
        if (exportFormat instanceof ExportFormatMulti) {
            ExportFormatMulti multi = (ExportFormatMulti) exportFormat;
            ExportFormat[] formats = multi.allExportFormats();
            String name = multi.formatName();
            CommandMenu menu11 = new CommandMenu(name);
            CommandMenu menuNN = new CommandMenu(name);
            CommandMenu menuN1 = new CommandMenu(name);
            for (ExportFormat f : formats) {
                buildExportFormat(f, menu11, menuNN, menuN1);
            }
            parent11.add(menu11);
            parent11.addSeparator();
            parentNN.add(menuNN);
            parentNN.addSeparator();
            parentN1.add(menuN1);
            parentN1.addSeparator();
        } else {
            generateCommands(exportFormat, parent11, parentNN, parentN1);
        }
    }

    /**
     * @require format != null
     * @require menu != null
     * @param format Export format
     * @param menu11 The menu from 1 to 1
     * @param menuNN The menu from n to n
     * @param menuN1 The menu from n to 1
     */
    private void generateCommands(
        ExportFormat format, CommandMenu menu11, CommandMenu menuNN, CommandMenu menuN1)
    {
        // 1 to 1
        {
            ExportFormatCommand command11 = generateExportCommand1to1(format);
            if (format.getShortCut() == -1) {
                menu11.add(command11);
            } else if (format.getModifier() == -1) {
                menu11.add(command11, format.getShortCut());
            } else {
                menu11.add(command11, format.getShortCut(), format.getModifier());
            }
        }
        // n to n
        {
            ExportFormatCommand commandNN = generateExportCommandNtoN(format);
            menuNN.add(commandNN);
        }

        // n to 1
        if (format.canExportNto1()) {
            ExportFormatCommand commandN1 = generateExportCommandNto1(format);
            menuN1.add(commandN1);
        }
    }

    /**
     * @require format != null
     * @ensure result != null
     * @param format Export format
     * @return The export command
     */
    private ExportFormatCommand generateExportCommand1to1(ExportFormat format) {
        ExportFormatCommand result = null;
        if (format != null) {
            result = new ExportFormatCommand(format, APPENDIX_11) {
                @Override
                public void execute() {
                    Drawing drawing = application().drawing();
                    if (drawing != null) {
                        if (!(drawing instanceof NullDrawing)) {
                            if (format().forceGivenName()) {
                                if (drawing.getFilename() != null) {
                                    try {
                                        String fileNameText =
                                            drawing.getFilename().getCanonicalPath()
                                                + drawing.getName();
                                        fileNameText = StringUtil.getPath(fileNameText)
                                            + File.separator + drawing.getName();
                                        File file = new File(fileNameText);
                                        saveDrawing(drawing, format(), file);
                                    } catch (IOException e) {
                                        logger.error("Could not create export file: ");
                                        logger.debug("Could not create export file: ", e);
                                    }
                                }
                            } else {
                                File path =
                                    DrawPlugin.getCurrent().getIOHelper()
                                        .getSaveFile(
                                            null,
                                            new SimpleFileFilter[] {
                                                (SimpleFileFilter) format().fileFilter() },
                                            drawing);
                                if (path != null) {
                                    displayer().showStatus("Exporting " + path + " ...");
                                    saveDrawing(drawing, format(), path);
                                }
                            }
                        } else {
                            displayer().showStatus("no drawing");
                        }
                    } else {
                        displayer().showStatus("no drawing");
                    }
                    application().toolDone();
                    //displayer().showStatus("export");
                }

                @Override
                public boolean isExecutable() {
                    return application() != null
                        && !(application().drawing() instanceof NullDrawing)
                        && format().canExportDrawing(application().drawing());
                }
            };
        }
        assert (result != null) : "Failure in GuiPlugin: result == null";
        return result;
    }

    /**
     * @require format != null
     * @ensure result != null
     * @param format The export format
     * @return The export command
     */
    private ExportFormatCommand generateExportCommandNto1(ExportFormat format) {
        ExportFormatCommand result = null;
        if (format != null) {
            result = new ExportFormatCommand(format, APPENDIX_N1) {
                @Override
                public void execute() {
                    Enumeration<Drawing> drawings = application().drawings();
                    if (drawings != null) {
                        if (drawings.hasMoreElements()) {
                            File path = DrawPlugin.getCurrent().getIOHelper().getSaveFile(
                                null,
                                new SimpleFileFilter[] { (SimpleFileFilter) format().fileFilter() },
                                drawings.nextElement());
                            if (path != null) {
                                displayer().showStatus("Exporting " + path + " ...");
                                saveDrawings(application().drawings(), format(), path, displayer());
                            }
                        } else {
                            displayer().showStatus("no drawing");
                        }
                    }
                    application().toolDone();
                    displayer().showStatus("export");
                }

                @Override
                public boolean isExecutable() {
                    if (application() == null || application().drawing() instanceof NullDrawing) {
                        return false;
                    }

                    Enumeration<Drawing> drawings = application().drawings();
                    while (drawings.hasMoreElements()) {
                        Drawing drawing = drawings.nextElement();
                        if (!format().canExportDrawing(drawing)) {
                            return false;
                        }
                    }

                    return true;
                }
            };
        }
        assert (result != null) : "Failure in GuiPlugin: result == null";
        return result;
    }

    /**
    * @require format != null
    * @ensure result != null
    * @param format The export format
    * @return The export format
    */
    private ExportFormatCommand generateExportCommandNtoN(ExportFormat format) {
        ExportFormatCommand result = null;
        if (format != null) {
            result = new ExportFormatCommand(format, APPENDIX_NN) {
                @Override
                public void execute() {
                    displayer().showStatus("Exporting...");
                    Enumeration<Drawing> drawings = application().drawings();
                    if (drawings != null) {
                        if (drawings.hasMoreElements()) {
                            saveDrawings(drawings, format());
                        } else {
                            displayer().showStatus("no drawing");
                        }
                    }
                    application().toolDone();
                    displayer().showStatus("export");
                }

                @Override
                public boolean isExecutable() {
                    if (application() == null || application().drawing() instanceof NullDrawing) {
                        return false;
                    }

                    Enumeration<Drawing> drawings = application().drawings();
                    while (drawings.hasMoreElements()) {
                        Drawing drawing = drawings.nextElement();
                        if (!format().canExportDrawing(drawing)) {
                            return false;
                        }
                    }
                    return true;
                }
            };
        }
        assert (result != null) : "Failure in GuiPlugin: result == null";
        return result;
    }

    @Override
    public void addExportFormat(ExportFormat exportFormat) {
        logger.debug(getClass() + ": adding export format " + exportFormat);
        exportFormats().add(exportFormat);
        buildExportFormat(exportFormat, exportMenu11(), exportMenuNN(), exportMenuN1());
    }

    @Override
    public ExportFormat[] allExportFormats() {
        return exportFormats().toArray(new ExportFormat[0]);
    }

    @Override
    public void removeExportFormat(ExportFormat format) {

        Component[] ele = exportMenu.getMenuComponents();
        for (Component component : ele) {
            if (component instanceof JMenuItem item) {
                if (item.getText().equals(format.formatName())) {
                    exportMenu.remove(item);
                }
            }
        }
        removeExportFormatFromMenu(format, _exportMenu11, APPENDIX_11);
        removeExportFormatFromMenu(format, _exportMenuN1, APPENDIX_N1);
        removeExportFormatFromMenu(format, _exportMenuNN, APPENDIX_NN);
        exportFormats().remove(format);
    }

    private void removeExportFormatFromMenu(
        ExportFormat format, CommandMenu menu, String appendix)
    {

        Component[] ele = menu.getMenuComponents();
        for (Component component : ele) {
            if (component instanceof JMenuItem item) {
                if (item.getText().equals(format.formatName() + appendix)) {
                    menu.remove(item);
                }
            }
        }

    }

}