package CH.ifa.draw.standard;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;
import java.util.List;
import javax.imageio.ImageIO;
import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import bibliothek.gui.dock.common.action.*;
import bibliothek.gui.dock.common.action.panel.PanelPopupWindow;
import bibliothek.gui.dock.common.intern.action.CDecorateableAction;

import CH.ifa.draw.DrawPlugin;
import CH.ifa.draw.figures.AttributeFigure;
import CH.ifa.draw.figures.PolyLineFigure;
import CH.ifa.draw.figures.TextFigure;
import CH.ifa.draw.framework.AlphaChangeCommand;
import CH.ifa.draw.framework.DrawingViewDecoration;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.util.ColorMap;
import CH.ifa.draw.util.Command;
import CH.ifa.draw.util.ExtendedFont;
import CH.ifa.draw.util.GUIProperties;


/**
 * Creates menus for changing the attributes of figures.
 * Can be used to decorate DrawingViews.
 */
public class AttributeMenuDecoration implements SelectionChangeListener, DrawingViewDecoration {

    private static final String IMAGES = "/CH/ifa/draw/images/";
    private static final List<String> basicFonts = new ArrayList<String>() {
        {
            add("Courier New");
            add("Andale Mono");
            add("Monaco");
            add("Profont");
            add("Monofur");
            add("Droid Sans Mono");
            add("DejaVu Sans Mono");
            add("Consolas");
            add("Inconsolata");
            add("Arial");
            add("Calibri");
            add("Lucida Sans");
        }
    };


    List<SelectionChangeListener> fSelectionChangeListeners =
        new LinkedList<SelectionChangeListener>();

    private LinkedList<CAction> menus;
    private int MAX_FONT_MENU_ENTRIES = 15;

    public LinkedList<CAction> getDecorations() {
        if (menus == null) {
            menus = createMenus();
        }

        return menus;
    }

    /**
     * Used to lazily create the decorations provided by this class.
     * The menus are saved for later reuse, so we don't create a lot of unnecessary objects.
     */
    private LinkedList createMenus() {
        LinkedList result = new LinkedList();
        result.add(createColorMenu("Fill Color", "FillColor", "fill_color"));
        result.add(
            createTransparencyMenu(
                "Fill Color Transparency", "FillColor", "transparency_fill.png"));
        result.add(createColorMenu("Pen Color", "FrameColor", "pen_color"));
        result.add(
            createTransparencyMenu("Pen Color Transparency", "FrameColor", "transparency_pen.png"));
        result.add(createArrowMenu());
        result.add(createLineStyleMenu());
        result.add(createLineShapeMenu());
        result.add(createLineWidthMenu());

        result.add(CSeparator.SEPARATOR);
        result.add(createColorMenu("Text Color", "TextColor", "text_color"));
        result.add(
            createTransparencyMenu(
                "Text Color Transparency", "TextColor", "transparency_text.png"));
        result.add(createFontSizeMenu());
        result.add(createFontStyleMenu());
        result.add(createTextAlignmentMenu());

        return result;
    }

    private CAction createTextAlignmentMenu() {
        CMenu menu = new CMenu();
        CommandExecutor button;
        ImageIcon icon;
        Command command;

        menu.setTooltip("Text Alignment");

        command = new ChangeAttributeCommand(
            "Left", TextFigure.ALIGN_ATTR, Integer.valueOf(TextFigure.LEFT));
        button = new CommandExecutor(command);
        button.setTooltip("Align Left");
        icon = new ImageIcon(loadImage(IMAGES + "textalign_left.png"));
        button.setIcon(icon);
        menu.add(button);

        command = new ChangeAttributeCommand(
            "Center", TextFigure.ALIGN_ATTR, Integer.valueOf(TextFigure.CENTER));
        button = new CommandExecutor(command);
        button.setTooltip("Align Center");
        icon = new ImageIcon(loadImage(IMAGES + "textalign_center.png"));
        button.setIcon(icon);
        menu.setIcon(icon);
        menu.add(button);

        command = new ChangeAttributeCommand(
            "Right", TextFigure.ALIGN_ATTR, Integer.valueOf(TextFigure.RIGHT));
        button = new CommandExecutor(command);
        button.setTooltip("Align Right");
        icon = new ImageIcon(loadImage(IMAGES + "textalign_right.png"));
        button.setIcon(icon);
        menu.add(button);

        setEnabledIfCommandExecutable(menu, command);

        return menu;
    }


    private CAction createFontStyleMenu() {
        CMenu menu = new CMenu();
        ImageIcon icon = new ImageIcon(loadImage(IMAGES + "TEXT.gif"));
        menu.setIcon(icon);
        menu.setTooltip("Font Style");
        menu.add(createFontStyleButton("Plain", Font.PLAIN, "TEXT.gif"));
        menu.add(createFontStyleButton("Italic", Font.ITALIC, "italic.png"));
        menu.add(createFontStyleButton("Bold", Font.BOLD, "bold.png"));
        menu.add(createFontStyleButton("Underlined", ExtendedFont.UNDERLINED, "underlined.png"));
        menu.addSeparator();
        menu.add(createFontMenu());

        setEnabledIfCommandExecutable(
            menu, new ChangeAttributeCommand("", "FontStyle", Font.PLAIN));

        return menu;
    }

    private CAction createFontStyleButton(String title, Integer type, String iconpath) {

        ChangeAttributeCommand command = new ChangeAttributeCommand(title, "FontStyle", type);
        CommandExecutor button = new CommandExecutor(command);

        button.setTooltip("Font Style: " + title);
        ImageIcon icon = new ImageIcon(loadImage(IMAGES + iconpath));
        button.setIcon(icon);
        button.setShowTextOnButtons(false);

        return button;
    }

    private CAction createFontSizeMenu() {
        final CPanelPopup popup = new CPanelPopup();
        popup.setCloseOnFocusLost(true);
        popup.setTooltip("Font Size");

        Integer[] sizes = { 9, 10, 11, 12, 14, 18, 24, 36, 48, 72 };
        JComboBox comboBox = new JComboBox(sizes);
        comboBox.setEditable(true);
        comboBox.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                JComboBox box = (JComboBox) e.getSource();
                new ChangeAttributeCommand("", "FontSize", box.getSelectedItem()).execute();
            }
        });

        popup.setContent(comboBox);
        ImageIcon icon = new ImageIcon(loadImage(IMAGES + "fontsize.png"));
        popup.setIcon(icon);

        setEnabledIfCommandExecutable(
            popup, new ChangeAttributeCommand("", "FontSize", Integer.valueOf(10)));
        return popup;
    }

    private CAction createFontMenu() {
        final CMenu menu = new CMenu();
        String[] fonts =
            GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames();
        CommandExecutor button;
        menu.setTooltip("Font Menu");
        menu.setText("Select Font");
        menu.setShowTextOnButtons(true);

        String defaultFontName = GUIProperties.defaultFontName();
        button = new CommandExecutor(
            new ChangeAttributeCommand(
                defaultFontName + " (default)", "FontName", defaultFontName));
        menu.add(button);

        int k = 0;
        for (int i = 0; (i < fonts.length) && (k < MAX_FONT_MENU_ENTRIES); i++) {

            if (basicFonts.contains(fonts[i]) && fonts[i] != defaultFontName) {
                button =
                    new CommandExecutor(new ChangeAttributeCommand(fonts[i], "FontName", fonts[i]));
                menu.add(button);
                k++;
            }

        }
        button = new CommandExecutor(
            new ChooseFontCommand("Font Name", "other...", "FontName", String.class));
        menu.add(button);

        setEnabledIfCommandExecutable(
            menu, new ChangeAttributeCommand(fonts[0], "FontName", fonts[0]));

        return menu;
    }

    private CAction createLineWidthMenu() {
        final CPanelPopup popup = new CPanelPopup();

        JSlider slider = new JSlider(JSlider.HORIZONTAL, 1, 10, 10);
        slider.setMajorTickSpacing(1);
        slider.setPaintLabels(true);
        slider.addChangeListener(new ChangeListener() {
            @Override
            public void stateChanged(ChangeEvent e) {
                JSlider s = (JSlider) e.getSource();
                if (!s.getValueIsAdjusting()) {
                    int value = s.getValue();
                    new SetLineWidthCommand("", value).execute();
                }

            }
        });

        popup.setCloseOnFocusLost(true);
        popup.setContent(slider);
        popup.setTooltip("Line Width");

        ImageIcon icon = new ImageIcon(loadImage(IMAGES + "linewidth.png"));
        popup.setIcon(icon);
        setEnabledIfCommandExecutable(popup, new SetLineWidthCommand("", 0));

        return popup;
    }

    private CAction createLineShapeMenu() {
        CMenu menu = new CMenu();
        CommandExecutor button;
        Command command;
        menu.setTooltip("Line Shape");
        ImageIcon icon = new ImageIcon(loadImage(IMAGES + "lineshape_both.png"));
        menu.setIcon(icon);

        command = new ChangeAttributeCommand(
            "straight", "LineShape", Integer.valueOf(PolyLineFigure.LINE_SHAPE));
        button = new CommandExecutor(command);
        button.setTooltip("Straight");
        button.setText("Straight");
        icon = new ImageIcon(loadImage(IMAGES + "lineshape_straight.png"));
        button.setIcon(icon);
        menu.add(button);

        command = new SplineAttributeCommand("standard", "standard", 0);
        button = new CommandExecutor(command);
        button.setTooltip("Curved");
        button.setText("Curved");
        icon = new ImageIcon(loadImage(IMAGES + "lineshape_curved.png"));
        button.setIcon(icon);
        menu.add(button);

        setEnabledIfCommandExecutable(menu, command);

        return menu;
    }

    private CAction createLineStyleMenu() {
        CMenu menu = new CMenu();
        menu.setTooltip("Line Style");
        menu.setIcon(createIcon("line_styles.gif"));
        CommandExecutor button;
        Command command;

        command =
            new ChangeAttributeCommand("normal", "LineStyle", AttributeFigure.LINE_STYLE_NORMAL);
        button = new CommandExecutor(command);
        button.setIcon(createIcon("linestyle_normal.gif"));
        menu.add(button);

        command =
            new ChangeAttributeCommand("dotted", "LineStyle", AttributeFigure.LINE_STYLE_DOTTED);
        button = new CommandExecutor(command);
        button.setIcon(createIcon("linestyle_dotted.gif"));
        menu.add(button);

        command =
            new ChangeAttributeCommand("dashed", "LineStyle", AttributeFigure.LINE_STYLE_DASHED);
        button = new CommandExecutor(command);
        button.setIcon(createIcon("linestyle_dashed.gif"));
        menu.add(button);

        command = new ChangeAttributeCommand(
            "medium dashed", "LineStyle", AttributeFigure.LINE_STYLE_MEDIUM_DASHED);
        button = new CommandExecutor(command);
        button.setIcon(createIcon("linestyle_mediumdashed.gif"));
        menu.add(button);

        command = new ChangeAttributeCommand(
            "long dashed", "LineStyle", AttributeFigure.LINE_STYLE_LONG_DASHED);
        button = new CommandExecutor(command);
        button.setIcon(createIcon("linestyle_longdashed.gif"));
        menu.add(button);

        command = new ChangeAttributeCommand(
            "dash-dotted", "LineStyle", AttributeFigure.LINE_STYLE_DASH_DOTTED);
        button = new CommandExecutor(command);
        button.setIcon(createIcon("linestyle_dashdotted.gif"));
        menu.add(button);

        setEnabledIfCommandExecutable(menu, command);

        command = new QueryAttributeCommand(
            "Line style (sequence of dash/gap lengths)", "other...", "LineStyle", String.class);
        button = new CommandExecutor(command);
        button.setIcon(createIcon("line_styles.gif"));
        menu.add(button);

        return menu;
    }

    private CAction createArrowMenu() {
        CDropDownButton menu = new CDropDownButton();
        menu.setIcon(createIcon("arrow_end.png"));
        menu.setTooltip("Arrows");
        CommandExecutor button;
        Command command;

        command = new ChangeAttributeCommand(
            "none", "ArrowMode", Integer.valueOf(PolyLineFigure.ARROW_TIP_NONE));
        button = new CommandExecutor(command);
        button.setIcon(createIcon("arrow_none.png"));
        menu.add(button);

        command = new ChangeAttributeCommand(
            "at Start", "ArrowMode", Integer.valueOf(PolyLineFigure.ARROW_TIP_START));
        button = new CommandExecutor(command);
        button.setIcon(createIcon("arrow_start.png"));
        menu.add(button);

        command = new ChangeAttributeCommand(
            "at End", "ArrowMode", Integer.valueOf(PolyLineFigure.ARROW_TIP_END));
        button = new CommandExecutor(command);
        button.setIcon(createIcon("arrow_end.png"));
        menu.add(button);

        command = new ChangeAttributeCommand(
            "at Start/End", "ArrowMode", Integer.valueOf(PolyLineFigure.ARROW_TIP_BOTH));
        button = new CommandExecutor(command);
        button.setIcon(createIcon("arrow_both.png"));
        menu.add(button);

        setEnabledIfCommandExecutable(menu, command);

        return menu;
    }

    private CAction createTransparencyMenu(String title, final String attribute, String iconpath) {

        final TransparencySlider slider = new TransparencySlider();

        slider.addChangeListener(new ChangeListener() {
            @Override
            public void stateChanged(ChangeEvent e) {
                if (!slider.getValueIsAdjusting()) {
                    int value = slider.getValue();
                    new AlphaChangeCommand("", attribute, (int) Math.round(value * 2.55)).execute();

                    slider.adjustLabels(value);
                }
            }
        });

        final CPanelPopup popup = new TransparencySliderCPanelPopup(attribute, slider);
        popup.setCloseOnFocusLost(true);
        popup.setTooltip(title);
        popup.setMenuBehavior(CPanelPopup.MenuBehavior.SUBMENU);

        ImageIcon icon = new ImageIcon(loadImage(IMAGES + iconpath));
        popup.setIcon(icon);

        setEnabledIfCommandExecutable(popup, new AlphaChangeCommand("", attribute, 0));

        return popup;
    }

    private CAction createColorMenu(String title, String attribute, String iconname) {
        CDropDownButton menu = new CDropDownButton();
        ImageIcon icon = new ImageIcon(loadImage(IMAGES + iconname + "_base.png"));
        menu.setIcon(icon);
        menu.setTooltip(title);
        CommandExecutor executor;
        Color color;


        for (int i = 0; i < ColorMap.size(); i++) {

            color = ColorMap.color(i);
            executor =
                new CommandExecutor(new ChangeAttributeCommand(ColorMap.name(i), attribute, color));

            BufferedImage img = loadImage(IMAGES + iconname + "_color.png");
            if (img != null) {
                changeColor(img, color);
                icon = new ImageIcon(img);
                executor.setIcon(icon);
            }
            menu.add(executor);

        }

        executor =
            new CommandExecutor(new ChooseColorCommand(title, "other...", attribute, Color.class));
        menu.add(executor);

        setEnabledIfCommandExecutable(menu, new ChangeAttributeCommand("", attribute, Color.BLUE));

        return menu;
    }

    @Override
    public void selectionStateChanged() {
        for (SelectionChangeListener listener : fSelectionChangeListeners) {
            listener.selectionStateChanged();
        }
    }

    /**
     * A {@link CButton} that executes a {@link Command} when pressed.
     */
    private class CommandExecutor extends CButton {

        final Command fCommand;

        public CommandExecutor(Command command) {
            fCommand = command;
            this.setText(command.name());
            this.setShowTextOnButtons(true);
            this.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    if (fCommand.isExecutable()) {
                        fCommand.execute();
                    }
                }
            });
        }
    }

    /**
     * A {@link CPanelPopup} that adjusts it's contained slider when opened.
     */
    private class TransparencySliderCPanelPopup extends CPanelPopup {

        final TransparencySlider slider;
        final String attribute;

        public TransparencySliderCPanelPopup(String attribute, TransparencySlider slider) {
            super();
            this.attribute = attribute;
            this.slider = slider;
            setContent(slider);
        }

        @Override
        public void openPopup(PanelPopupWindow window) {
            int value = 100;
            DrawPlugin plugin = DrawPlugin.getCurrent();
            if (plugin.isGuiPresent()) {
                Vector<Figure> v = plugin.getGui().view().selection();
                if (v.size() > 0) {
                    Figure f = v.get(0);
                    Color color = (Color) f.getAttribute(attribute);
                    double alpha = color.getAlpha() / 2.55;
                    value = (int) alpha;
                }
            }

            // This is kind of hacky, but probably the simplest solution.
            ChangeListener[] listeners = slider.getChangeListeners();
            slider.removeChangeListener(listeners[0]);
            slider.setValue(value);
            slider.adjustLabels(value);
            slider.addChangeListener(listeners[0]);

            super.openPopup(window);
        }
    }

    /**
     * A {@link JSlider} used to adjust transparency.
     * Provides a method to adjust the labels.
     */
    private class TransparencySlider extends JSlider {

        public TransparencySlider() {
            super(JSlider.HORIZONTAL, 0, 100, 100);
            this.setMajorTickSpacing(20);
            this.setMinorTickSpacing(10);
            this.setPaintTicks(true);
            this.setPaintLabels(true);
        }

        public void adjustLabels(int value) {
            // Create new labels so we can show the current value
            Hashtable labels = this.createStandardLabels(20);
            labels.put(Integer.valueOf(value), new JLabel("" + value));

            // Remove the closest regular label so they don't overlap
            double valueAsDouble = (double) value;
            int toRemove = (int) Math.round(valueAsDouble / 20) * 20;
            labels.remove(toRemove);
            this.setLabelTable(labels);
        }
    }

    /**
     * Helper method to create icons.
     *
     * @param name
     * @return
     */
    private ImageIcon createIcon(String name) {
        BufferedImage img = loadImage(IMAGES + name);
        return new ImageIcon(img);
    }

    /**
     * Tries to load an image from a pathname.
     * Used for later creating icon images.
     */
    private BufferedImage loadImage(final String path) {

        BufferedImage result = null;

        URL url = AttributeMenuDecoration.class.getResource(path);
        if (url == null) {
            File file = new File(path);
            if (file != null) {
                try {
                    url = file.toURI().toURL();
                } catch (MalformedURLException e) {
                    e.printStackTrace();
                }
            }
        }
        if (url != null) {
            try {
                result = ImageIO.read(url);
                return result;
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        return result;
    }

    private void setEnabledIfCommandExecutable(
        final CDecorateableAction toggable, Command command)
    {

        final Command commandCopy = command;

        fSelectionChangeListeners.add(new SelectionChangeListener() {

            // The command is never executed.
            // It is solely used to see if the Action is activated.

            @Override
            public void selectionStateChanged() {
                if (!commandCopy.isExecutable()) {
                    toggable.setEnabled(false);
                } else {
                    toggable.setEnabled(true);
                }
            }
        });
    }


    /**
     * Changes the color of an icon.
     * This method iterates over every pixel, so only use on small icon images.
     */
    private void changeColor(BufferedImage img, Color color) {
        Color standard = new Color(255, 0, 255);
        changeColor(img, standard, color);
    }

    /**
     * Changes the color of an icon.
     * This method iterates over every pixel, so only use on small icon images.
     */
    private void changeColor(BufferedImage img, Color from, Color to) {
        final int oldRGB = from.getRGB();
        final int newRGB = to.getRGB();
        for (int x = 0; x < img.getWidth(); x++) {
            for (int y = 0; y < img.getHeight(); y++) {
                if (img.getRGB(x, y) == oldRGB)
                    img.setRGB(x, y, newRGB);
            }
        }
    }
}
