package CH.ifa.draw.standard;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;

import CH.ifa.draw.framework.Drawing;
import CH.ifa.draw.framework.DrawingView;
import CH.ifa.draw.util.DrawingListener;


public class StandardDrawingLookup {

    public static org.apache.log4j.Logger logger =
        org.apache.log4j.Logger.getLogger(StandardDrawingLookup.class);

    public DrawingView getPreviousView() {
        return fPreviousView;
    }

    private DrawingView fPreviousView;
    private DrawingView fCurrentView;
    private Map<Drawing, StandardDrawingViewContainer> fOpenedDrawings =
        new HashMap<Drawing, StandardDrawingViewContainer>();
    private Vector<DrawingListener> fDrawingListeners = new Vector<DrawingListener>();

    /**
     * Set the currently selected {@link DrawingView}
     *
     * @param currentView
     */
    public void setCurrentView(DrawingView currentView) {

        logger.debug("Setting current view to " + currentView.drawing().getName());

        this.fPreviousView = this.fCurrentView;
        this.fCurrentView = currentView;
    }

    /**
     * @return the currently focused {@link DrawingView}
     */
    public DrawingView getCurrentView() {
        if (fCurrentView != null) {
            return fCurrentView;
        }
        return NullDrawingView.INSTANCE;
    }

    /**
     * Returns the String identifier of the view container wrapping this drawing in the gui.
     *
     * @param drawing
     * @return A String object identifying a view container in the gui
     */
    public StandardDrawingViewContainer getViewContainerForDrawing(Drawing drawing) {

        try {
            return fOpenedDrawings.get(drawing);
        } catch (Exception e) {
            logger.error("Tried to retrieve a view container for a drawing that isn't opened.");
        }
        return null;
    }

    /**
     * Add a {@link Drawing} and it's view container to the set of currently tracked drawings,
     * or replaces the associated view container of a {@link Drawing}.
     *
     * @param drawing
     * @param container
     */
    public void addViewContainer(Drawing drawing, StandardDrawingViewContainer container) {
        fOpenedDrawings.put(drawing, container);
        drawingAdded(drawing);
    }

    /**
     * Removes the {@link Drawing} and it's associated view container from the set of tracked entities.
     *
     * @param drawing
     */
    public void removeViewContainer(Drawing drawing) {
        fOpenedDrawings.remove(drawing);
        drawingReleased(drawing);
    }


    /**
     * Check whether a {@link Drawing} saved in a {@link File} is already loaded.
     *
     * @param file A saved {@link Drawing} file.
     * @return If the file is opened the respective drawing, null otherwise
     */
    public Drawing getLoadedDrawingForFile(File file) {
        for (Map.Entry<Drawing, StandardDrawingViewContainer> entry : fOpenedDrawings.entrySet()) {
            File filename = entry.getKey().getFilename();
            if (filename != null && filename.getAbsoluteFile().equals(file.getAbsoluteFile())) {
                return entry.getKey();
            }
        }
        return null;
    }


    /**
     * Get all currently known drawings as an {@link Enumeration}.
     *
     * @return All open drawings
     */
    public Enumeration<Drawing> getAllDrawings() {

        Vector<Drawing> vector = new Vector<Drawing>(fOpenedDrawings.keySet());
        return vector.elements();
    }


    /**
     * Removes all currently known drawings.
     * Users will be asked if they want to save changed drawings and may cancel this operation.
     */
    public void closeAllDrawings() {

        logger.debug("Closing all opened drawings.");

        ArrayList<Drawing> allDrawings = Collections.list(getAllDrawings());
        // sort by descending name to close instances before templates
        allDrawings.sort((d1, d2) -> d2.getName().compareTo(d1.getName()));

        for (Drawing drawingEntry : allDrawings) {
            StandardDrawingViewContainer container = fOpenedDrawings.get(drawingEntry);
            // drawing might have been closed concurrently
            if (container != null) {
                Drawing drawing = container.getView().drawing();
                container.setVisible(false);

                // If container is still visible, operation was canceled by user.
                if (!container.isVisible()) {

                    fOpenedDrawings.remove(drawing);
                    drawingReleased(drawing);
                }
            }
        }
    }

    /**
     * Adds drawing listener to the list of drawing listeners
     *
     * @param listener the listener to add
     */
    public void addDrawingListener(DrawingListener listener) {
        fDrawingListeners.add(listener);
    }

    /**
     * Removes drawing listener from the list of drawing listeners.
     *
     * @param listener the listener to remove
     */
    public void removeDrawingListener(DrawingListener listener) {
        fDrawingListeners.remove(listener);
    }

    /**
     * Inform listeners that a drawing was added
     * @param drawing the added drawing
     */
    private void drawingAdded(Drawing drawing) {
        fDrawingListeners.forEach(listener -> listener.drawingAdded(drawing));
    }

    /**
     * Informs listeners that a drawing was released and removes it from
     * stored views.
     * @param drawing the released drawing
     */
    private void drawingReleased(Drawing drawing) {
        fDrawingListeners.forEach(listener -> listener.drawingReleased(drawing));

        if (fPreviousView != null && fPreviousView.drawing().equals(drawing)) {
            fPreviousView = NullDrawingView.INSTANCE;
        }
        if (fCurrentView != null && fCurrentView.drawing().equals(drawing)) {
            fCurrentView = fPreviousView;
            fPreviousView = NullDrawingView.INSTANCE;
        }

    }
}
