package CH.ifa.draw.standard;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import bibliothek.gui.dock.common.DefaultMultipleCDockable;
import bibliothek.gui.dock.common.MultipleCDockableFactory;


public class StandardDrawingViewContainer extends DefaultMultipleCDockable {

    private final JScrollPane fScrollPane;

    public StandardDrawingView getView() {
        return fView;
    }

    private StandardDrawingView fView;
    private static final int VERTICAL_SCROLL_INCREMENT = 75;
    private static final int HORIZONTAL_SCROLL_INCREMENT = 75;

    public StandardDrawingViewContainer(
        MultipleCDockableFactory<?, ?> factory, StandardDrawingView view)
    {
        super(factory);

        this.setCloseable(true);
        this.setRemoveOnClose(true);
        fView = view;

        InputEventForwarder forwarder = new InputEventForwarder();
        view.addMouseListener(forwarder);
        view.addMouseMotionListener(forwarder);
        view.addKeyListener(forwarder);

        fScrollPane = new JScrollPane(
            view, ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
            ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
        fScrollPane.getViewport().setScrollMode(javax.swing.JViewport.BLIT_SCROLL_MODE);

        //Let the user scroll by dragging to outside the window.
        fScrollPane.setAutoscrolls(true); //enable synthetic drag events
        fScrollPane.getVerticalScrollBar().setUnitIncrement(VERTICAL_SCROLL_INCREMENT);
        fScrollPane.getHorizontalScrollBar().setUnitIncrement(HORIZONTAL_SCROLL_INCREMENT);
        fScrollPane.addMouseWheelListener(forwarder);

        this.add(fScrollPane);
    }

    public JScrollPane getScrollPane() {
        return fScrollPane;
    }

    /**
     * Performs a cleanup of this container.
     * This will also remove the view reference.
     */
    public void clean() {

        getView().getParent().remove(getView());
        fView = null;

    }

    /**
     * An instance of this class is used to catch all input (mouse and
     * keyboard) events and forward them to the StandardDrawingView.
     **/
    private class InputEventForwarder
        implements KeyListener, MouseMotionListener, MouseListener, MouseWheelListener
    {
        private boolean active;

        public InputEventForwarder() {
            this.active = true;
        }

        // implementation of java.awt.event.MouseListener interface
        public void mouseClicked(MouseEvent e) {
            if (active) {
                fView.mouseClicked(e);
            }
        }

        public void mousePressed(MouseEvent e) {
            if (active) {
                fView.mousePressed(e);
            }
        }

        public void mouseReleased(MouseEvent e) {
            if (active) {
                fView.mouseReleased(e);
            }
        }

        public void mouseEntered(MouseEvent e) {
            if (active) {
                fView.mouseEntered(e);
            }
        }

        public void mouseExited(MouseEvent e) {
            if (active) {
                fView.mouseExited(e);
            }
        }

        // implementation of java.awt.event.MouseMotionListener interface
        public void mouseDragged(MouseEvent e) {
            if (active) {
                Rectangle r = new Rectangle(e.getX(), e.getY(), 1, 1);
                ((JPanel) e.getSource()).scrollRectToVisible(r);
                fView.mouseDragged(e);
            }
        }

        public void mouseMoved(MouseEvent e) {
            if (active) {
                fView.mouseMoved(e);
            }
        }

        // implementation of java.awt.event.KeyListener interface
        public void keyTyped(KeyEvent e) {
            if (active) {
                fView.keyTyped(e);
            }
        }

        public void keyPressed(KeyEvent e) {
            if (active) {
                fView.keyPressed(e);
            }
            if (active) {

                if (e.getKeyCode() == KeyEvent.VK_CONTROL || e.getKeyCode() == KeyEvent.VK_META
                    || e.getKeyCode() == Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()) {
                    fScrollPane.setWheelScrollingEnabled(false);
                }
            }

        }

        public void keyReleased(KeyEvent e) {
            if (active) {
                fView.keyReleased(e);
            }
            if (active) {

                if (e.getKeyCode() == KeyEvent.VK_CONTROL || e.getKeyCode() == KeyEvent.VK_META
                    || e.getKeyCode() == Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()) {
                    fScrollPane.setWheelScrollingEnabled(true);
                }
            }

        }

        public void discard() {
            active = false;
        }

        @Override
        public void mouseWheelMoved(MouseWheelEvent e) {
            if (e.getModifiers() == Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()) {
                fView.mouseWheelMoved(e);

                // After zooming need to repaint, because the DrawingView might be smaller than the scrollpane
                fScrollPane.repaint();

            }
        }
    }
}
