package CH.ifa.draw.standard;

import java.util.Collections;
import java.util.List;
import java.util.Vector;
import java.util.stream.Collectors;

import CH.ifa.draw.DrawPlugin;
import CH.ifa.draw.framework.DrawingEditor;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureWithID;
import CH.ifa.draw.framework.SnapshotHistory;
import CH.ifa.draw.util.Command;


/**
 * Restores one undo or redo step.
 *
 * UndoRedoCommand.java
 * Created: Wed Dec 13  2000
 * @author Michael Duvigneau
 *
 * @see CH.ifa.draw.framework.UndoRedoManager
 */
public class UndoRedoCommand extends Command {
    public static enum Mode {
        UNDO, REDO
    };

    // private DrawingEditor editor;
    private Mode mode;

    public UndoRedoCommand(String name, Mode mode) {
        super(name);
        // this.editor = editor;
        this.mode = mode;
    }

    private DrawingEditor getEditor() {
        DrawPlugin plugin = DrawPlugin.getCurrent();
        return (plugin == null) ? NullDrawingEditor.INSTANCE : plugin.getDrawingEditor();
    }

    @Override
    public boolean isExecutable() {
        if (!super.isExecutable()) {
            return false;
        }
        if (getEditor() == NullDrawingEditor.INSTANCE) {
            return false;
        }
        SnapshotHistory history = null;
        switch (mode) {
            case UNDO:
                history = getEditor().getUndoRedoManager().getUndoHistory(getEditor().drawing());
                break;
            case REDO:
                history = getEditor().getUndoRedoManager().getRedoHistory(getEditor().drawing());
                break;
        }
        return (history != null) && (!history.isEmpty());
    }

    @Override
    public void execute() {
        if (isExecutable()) {
            Vector<Figure> selectionBefore = getEditor().view().selection();
            List<Integer> selectedIds =
                selectionBefore.stream().filter(FigureWithID.class::isInstance)
                    .map(FigureWithID.class::cast).map(f -> f.getID()).collect(Collectors.toList());

            getEditor().toolDone();
            switch (mode) {
                case UNDO:
                    getEditor().getUndoRedoManager().restoreUndoSnapshot(getEditor().drawing());
                    break;
                case REDO:
                    getEditor().getUndoRedoManager().restoreRedoSnapshot(getEditor().drawing());
                    break;
            }

            Vector<Figure> selectionAfter = Collections.list(getEditor().drawing().figures())
                .stream().filter(FigureWithID.class::isInstance).map(FigureWithID.class::cast)
                .filter(f -> selectedIds.contains(f.getID()))
                .collect(Collectors.toCollection(Vector::new));
            // reselect the figures if sizes of selections match (might not be 
            // true if ids are not unique).  
            if (selectionBefore.size() == selectionAfter.size()) {
                getEditor().view().addToSelectionAll(selectionAfter);
            }
            getEditor().selectionChanged(getEditor().view());
        }
    }
}
