package CH.ifa.draw.util;

import java.util.Hashtable;

import CH.ifa.draw.framework.Drawing;


/**
 * This class manages the autosaving of drawings. For each added drawing, a separate AutosaveTask
 * is created and the drawing is registered with that AutosaveTask.
 */
public class AutosaveManager {
    private Hashtable<Drawing, AutosaveTask> autosaveTasks = new Hashtable<Drawing, AutosaveTask>();
    private long interval;
    private AutosaveSaver saver;

    /**
     * Constructs a new autosave manager with an AutosaveSaver
     * and an interval determining the time between saves.
     *
     * @param saver the saver used to save drawings/files
     * @param interval the interval at which the drawings/files will be saved
     */
    public AutosaveManager(AutosaveSaver saver, long interval) {
        this.saver = saver;
        this.interval = interval;
    }

    /**
     * Includes a drawing in the list of drawings to be autosaved.
     * If the drawing is already included in the list, reset
     * its last save time to now.
     *
     * @param drawing the <code>Drawing</code> to be autosaved
     */
    public synchronized void addDrawing(Drawing drawing) {
        AutosaveTask task = autosaveTasks.get(drawing);
        if (task == null) {
            task = new AutosaveTask(saver, drawing, interval);
            autosaveTasks.put(drawing, task);
        }
        task.reset();
    }

    /**
     * Excludes a drawing from the list of drawings to be autosaved.
     * Remove its autosave file, if any.
     *
     * @param drawing the <code>Drawing</code> not to be autosaved
     */
    public synchronized void removeDrawing(Drawing drawing) {
        AutosaveTask task = autosaveTasks.get(drawing);
        if (task != null) {
            task.terminate();
            autosaveTasks.remove(drawing);
        }
    }

    /**
     * Update the name of the autosave file associated to a
     * drawing. This method should be called after a drawing has
     * been renamed. If the name of a drawing changes without
     * calling this method, the old autosave file will still be
     * used, which is allowed.
     *
     * @param drawing the <code>Drawing</code> to be renamed
     */
    public synchronized void renameDrawing(Drawing drawing) {
        removeDrawing(drawing);
        addDrawing(drawing);
    }
}