package de.renew.plugin;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertInstanceOf;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertSame;

public final class SoftDependencyTest {
    /**
     * <code>SoftDependencyListener</code> with a regular 1-arg constructor
     * (as specified in {@link SoftDependencyListener})
     */
    private static final String LISTENER_STRING =
        SoftDependencyTest.class.getCanonicalName() + "$DummySoftDependencyListener";
    /**
     * <code>SoftDependencyListener</code> with only a no-arg constructor
     */
    private static final String LISTENER_NOARG_STRING =
        SoftDependencyTest.class.getCanonicalName() + "$DummyNoArgSoftDependencyListener";

    private static IPlugin _caller;

    private SoftDependency _testee;

    @BeforeAll
    public static void setUp() {
        // Create a dummy plugin with no provisions and no dependencies
        final PluginProperties props = new PluginProperties();
        _caller = new PluginAdapter(props);
    }

    @Test
    void testGetListenerWithArgs() {
        _testee = new SoftDependency(_caller, "", LISTENER_STRING);
        final SoftDependencyListener result = _testee.getListener();

        assertNotNull(result);
        assertInstanceOf(DummySoftDependencyListener.class, result);

        // Consecutive calls should return the same object
        assertSame(result, _testee.getListener());
    }

    @Test
    void testGetListenerNoArgs() {
        _testee = new SoftDependency(_caller, "", LISTENER_NOARG_STRING);
        final SoftDependencyListener result = _testee.getListener();

        assertNotNull(result);
        assertInstanceOf(DummyNoArgSoftDependencyListener.class, result);

        // Consecutive calls should return the same object
        assertSame(result, _testee.getListener());
    }

    /*
     * Dummy listeners for use with the tests.
     * They do not actually provide any functionality.
     */

    public static class DummySoftDependencyListener implements SoftDependencyListener {

        // No no-arg constructor for this class
        public DummySoftDependencyListener(@SuppressWarnings("unused") IPlugin dummy) {}

        @Override
        public void serviceAvailable(IPlugin plugin) {}

        @Override
        public void serviceRemoved(IPlugin plugin) {}
    }

    public static class DummyNoArgSoftDependencyListener implements SoftDependencyListener {

        @Override
        public void serviceAvailable(IPlugin plugin) {}

        @Override
        public void serviceRemoved(IPlugin plugin) {}
    }
}
