package de.renew.lola2.parser;

import java.util.HashMap;
import java.util.Map;

/**
 * Represents a transition in a Lola2 Petri net.
 */
public class Transition extends Node {
    private Map<Place, Integer> _pre = new HashMap<Place, Integer>();
    private Map<Place, Integer> _post = new HashMap<Place, Integer>();

    /**
     * Default constructor for a transition with a name.
     *
     * @param name the name of the transition
     */
    protected Transition(String name) {
        super(name);
    }

    /**
     * Default constructor for a transition with a name, pre-conditions, and post-conditions.
     *
     * @param name the name of the transition
     * @param pre the pre-conditions (mapping of places to their required tokens)
     * @param post the post-conditions (mapping of places to the tokens added)
     */
    public Transition(String name, Map<Place, Integer> pre, Map<Place, Integer> post) {
        super(name);
        this._pre = pre;
        this._post = post;
    }

    /**
     * Default constructor for a transition with a name and coordinates.
     *
     * @param name the name of the transition
     * @param x the x-coordinate of the transition
     * @param y the y-coordinate of the transition
     */
    protected Transition(String name, int x, int y) {
        super(name, x, y);
    }

    /**
     * Default constructor for a transition with a name, coordinates, pre-conditions, and post-conditions.
     *
     * @param name the name of the transition
     * @param x the x-coordinate of the transition
     * @param y the y-coordinate of the transition
     * @param pre the pre-conditions (mapping of places to their required tokens)
     * @param post the post-conditions (mapping of places to the tokens added)
     */
    protected Transition(
        String name, int x, int y, Map<Place, Integer> pre, Map<Place, Integer> post)
    {
        super(name, x, y);
        this._pre = pre;
        this._post = post;
    }

    /**
     * Returns the pre-conditions of this transition.
     *
     * @return the amount of tokens required in the pre-places for this transition
     */
    protected Map<Place, Integer> getPre() {
        return _pre;
    }

    /**
     * Sets the pre-conditions of this transition.
     *
     * @param pre the amount of tokens required in the pre-places for this transition
     */
    protected void setPre(Map<Place, Integer> pre) {
        this._pre = pre;
    }

    /**
     * Returns the post-conditions of this transition.
     *
     * @return the amount of tokens added to the post-places after this transition
     */
    protected Map<Place, Integer> getPost() {
        return _post;
    }

    /**
     * Sets the post-conditions of this transition.
     *
     * @param post the amount of tokens added to the post-places after this transition
     */
    protected void setPost(Map<Place, Integer> post) {
        this._post = post;
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof Transition) {
            return getName().equals(((Transition) o).getName());
        } else {
            return false;
        }
    }
}