package de.renew.net;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import de.renew.application.SimulationRunningException;
import de.renew.engine.simulator.SimulationThreadPool;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertSame;
import static org.junit.jupiter.api.Assertions.assertThrows;

/**
 * This class is a test class for the interaction between the classes {@link Place}
 * and {@link Net} in the module de.renew.util.
 */
public class PlaceTest {
    Net net;
    Place place;
    Place otherPlace;
    Place thirdPlace;
    NetElementID placeNetElementID;
    int placeCount;
    boolean done;
    AssertionError thrown;

    /**
     * This method should be run before each test method.
     */
    @BeforeEach
    void setUp() {
        net = null;
        place = null;
        otherPlace = null;
        thirdPlace = null;
        placeNetElementID = null;
        done = false;
        thrown = null;
        placeCount = -1;
    }

    /**
     *Tests the interaction between the constructor of the class {@link Place}
     * and the method {@link Net#add(Place)}.
     * Tests the case of adding two places to the same net.
     * @throws InterruptedException
     */
    @Test
    void testAddNewPlace() throws InterruptedException {
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                try {
                    net = new Net("Test net");
                    NetElementID placeNetElementID = new NetElementID(1);
                    place = new Place(net, "Test Place", placeNetElementID);
                    otherPlace = new Place(net, "Other place", new NetElementID(2));
                    thirdPlace = net.getPlaceWithID(placeNetElementID);
                    placeCount = net.placeCount();
                    done = true;
                } catch (SimulationRunningException e) {
                    SimulationThreadPool.discardNew();
                    throw e;
                }
            }
        };
        runBySimulationThread(runnable);
        assertSame(place, thirdPlace);
        assertEquals(2, placeCount);
    }

    /**
     * Tests the interaction between the constructor of the class {@link Place}
     * and the method {@link Net#add(Place)} from the Net class.
     * Tests the case of adding two places with the same {@link NetElementID} to the same net.
     * In this case an {@link AssertionError} should be thrown.
     * @throws InterruptedException
     */
    @Test
    void testAddNewPlaceWithExistingID() throws InterruptedException {
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                try {
                    net = new Net("Test net");
                    placeNetElementID = new NetElementID();
                    place = new Place(net, "Old place", placeNetElementID);
                    thrown = assertThrows(AssertionError.class, () -> {
                        otherPlace = new Place(net, "New place", placeNetElementID);
                    });
                    placeCount = net.placeCount();
                    done = true;
                } catch (SimulationRunningException e) {
                    SimulationThreadPool.discardNew();
                    throw e;
                }
            }
        };
        runBySimulationThread(runnable);
        String message = "Tried to add place with existing ID: " + placeNetElementID + ", old: "
            + place + ", new: New place";
        assertEquals(message, thrown.getMessage());
        assertEquals(1, placeCount);
    }

    /**
     * Tests the interaction between the constructor of the class {@link Place}
     * and the method {@link Net#add(Place)} from the Net class.
     * Tests the case of adding the same place twice to the same net.
     * In this case an {@link AssertionError} should be thrown.
     * @throws InterruptedException
     */
    @Test
    void testAddExistingPlace() throws InterruptedException {
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                try {
                    net = new Net("Test net");
                    placeNetElementID = new NetElementID();
                    place = new Place(net, "Old place", placeNetElementID);
                    thrown = assertThrows(AssertionError.class, () -> {
                        net.add(place);
                    });

                    placeCount = net.placeCount();
                    done = true;
                } catch (SimulationRunningException e) {
                    SimulationThreadPool.discardNew();
                    throw e;
                }
            }
        };
        runBySimulationThread(runnable);

        String message = "Tried to add existing place: " + place;
        assertEquals(message, thrown.getMessage());
        assertEquals(1, placeCount);
    }

    /**
     * Runs a runnable by a simulation thread and waits until the execution is finished.
     * @param runnable
     * @throws InterruptedException
     */
    private void runBySimulationThread(Runnable runnable) throws InterruptedException {
        done = false;
        SimulationThreadPool.getNew();
        SimulationThreadPool simulationThreadPool = SimulationThreadPool.getSimulationThreadPool();
        Thread simulationThread = simulationThreadPool.getThreadFactory().newThread(runnable);
        simulationThread.start();
        while (!done) {
            System.out.println("Simulation thread is running");
            Thread.sleep(2000);
        }
    }
}