package de.renew.application;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.StreamCorruptedException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;

import de.renew.database.SetupHelper;
import de.renew.database.SetupHelper.SimulationState;
import de.renew.engine.searchqueue.SearchQueue;
import de.renew.engine.simulator.ConcurrentSimulator;
import de.renew.engine.simulator.InheritableSimulationThreadLock;
import de.renew.engine.simulator.NonConcurrentSimulator;
import de.renew.engine.simulator.ParallelSimulator;
import de.renew.engine.simulator.SimulationThreadPool;
import de.renew.engine.simulator.Simulator;
import de.renew.engine.simulator.SimulatorEventQueue;
import de.renew.net.IDRegistry;
import de.renew.net.Net;
import de.renew.net.NetInstance;
import de.renew.net.NetNotFoundException;
import de.renew.net.loading.Finder;
import de.renew.net.loading.NetLoader;
import de.renew.net.loading.PathlessFinder;
import de.renew.plugin.IPlugin;
import de.renew.plugin.PluginAdapter;
import de.renew.plugin.PluginManager;
import de.renew.plugin.PluginProperties;
import de.renew.plugin.PropertyHelper;
import de.renew.shadow.DefaultCompiledNetLoader;
import de.renew.shadow.DefaultShadowNetLoader;
import de.renew.shadow.SNSFinder;
import de.renew.shadow.SequentialOnlyExtension;
import de.renew.shadow.ShadowLookup;
import de.renew.shadow.ShadowNetLoader;
import de.renew.shadow.ShadowNetSystem;
import de.renew.shadow.SyntaxException;
import de.renew.util.ClassSource;
import de.renew.util.RenewObjectInputStream;
import de.renew.util.RenewObjectOutputStream;
import de.renew.util.SingletonException;


/**
 * This class serves as the main facade for the Renew simulator component. It can be
 * used to set up one Renew simulation per virtual machine. The simulation
 * engine can be configured and extended in various ways.
 * <p>
 * This class combines most of the functionality of the old classes
 * <code>de.renew.gui.CPNSimulation</code> and
 * <code>de.renew.application.ShadowSimulator</code>. The old subclasses of
 * <code>ShadowSimulator</code> (<code>de.renew.remote.ServerImpl</code>,
 * <code>de.renew.access.AccessControlledServerImpl</code> and
 * <code>de.renew.workflow.WorkflowServerImpl</code>) are now covered by the
 * extension interface defined along with this class.
 * </p>
 * <p>
 * This class must be used as a singleton (and in fact all constructors and
 * methods enforce this) because some parts of the simulation engine use global
 * states in static fields.
 * </p>
 * <p>
 * One way to set up a simulation might look as follows:
 *
 * <pre>
 *        // These things need to be known beforehand:
 *        String mainNet;                   // The name of the net for
 *                                          // the initial instance.
 *        ShadowNetSystem shadowNetSystem;  // All nets needed for the
 *                                          // simulation initially.
 *
 *        // Get the simulator plugin
 *        SimulatorPlugin simulatorPlugin = SimulatorPlugin.getCurrent();
 *
 *        // Allocate storage for primary net instance reference
 *        NetInstance primaryInstance = null;
 *
 *        // Acquire mutual exclusion for operations on simulator plug-in.
 *        simulatorPlugin.lock.lock();
 *        try {
 *            // Obtain fresh simulation pool thread to set up new simulation.
 *            Future&lt;NetInstance&gt; future = SimulationThreadPool.getNew().submitAndWait(new Callable&lt;NetInstance&gt;() {
 *                public NetInstance call() throws SomeException {
 *
 *                    // Use default net loader.
 *                    simulatorPlugin.setDefaultNetLoader();
 *
 *                    // Initialise the simulation.
 *                    simulatorPlugin.setupSimulation(null);
 *
 *                    // Compile and add nets.
 *                    simulatorPlugin.insertNets(shadowNetSystem);
 *
 *                    // Create the initial net instance.
 *                    NetInstance primaryInstance = simulatorPlugin.createNetInstance(mainNet));
 *
 *                    // Start the simulation.
 *                    simulatorPlugin.getCurrentEnvironment().getSimulator().startRun();
 *
 *                    return primaryInstance;
 *                }
 *            });
 *            primaryInstance = future.get();
 *        } catch (ExecutionException e) {
 *           ...
 *        } finally {
 *                 // Release the mutual exclusion lock under any circumstances!
 *                simulatorPlugin.lock.unlock();
 *        }
 * </pre>
 * <p>
 * SimulatorPlugin.java Created: Mon Jun 2 2003
 *
 * @author Michael Duvigneau
 * @since Renew 2.0
 */
public class SimulatorPlugin extends PluginAdapter implements SimulatorManager {
    /**
     * The logger for this class.
     */
    private static final Logger LOGGER = Logger.getLogger(SimulatorPlugin.class);

    /**
     * Used to synchronise access to the static <code>singleton</code> variable.
     */
    private static final Object SINGLETON_LOCK = new Object();

    /**
     * Holds a reference to the one and only SimulatorPlugin instance. Set by
     * the {@link #SimulatorPlugin constructor} and reset by the method
     * {@link #cleanup}. To check whether the current object is still the
     * current singleton instance, call {@link #checkSingleton} at each public
     * method entry point.
     */
    private static SimulatorPlugin _singleton = null;

    /**
     * Holds the set of all {@link SimulatorExtension} objects currently
     * registered with this plugin. Can be modified by {@link #addExtension} and
     * {@link #removeExtension}.
     */
    private final Queue<SimulatorExtension> _extensions = new ConcurrentLinkedQueue<>();

    /**
     * Holds all information about the current simulation environment.
     */
    private SimulationEnvironment _currentSimulation = null;

    /**
     * Flags that the current simulation is virgin, that no nets have been added
     * yet.
     */
    private boolean _virginSimulation;

    /**
     * Remembers the default shadow net loader for the current simulation if
     * one has been requested.
     */
    private DefaultShadowNetLoader _currentShadowNetLoader = null;

    /**
     * Holds the net loader to use in the next simulation setup.
     */
    private NetLoader _nextNetLoader = null;

    /**
     * Holds all registered finders for the default shadow net loader. Access to
     * this set is synchronized on the set itself, not on the
     * <code>SimulatorPlugin</code> instance.
     */
    private final Set<Finder> _registeredFinders = Collections.synchronizedSet(new HashSet<>());

    /**
     * Holds all registered pathless finders for the default shadow net loader.
     * Access to this set is synchronized on the set {@link #_registeredFinders},
     * not on the <code>SimulatorPlugin</code> instance or the set itself.
     */
    private final Set<PathlessFinder> _registeredPathlessFinders = new HashSet<>();

    /**
     * Remembers whether classReinit mode was active in the last run.
     */
    private boolean _previousClassReinit = false;

    /**
     * Holds a reference to the SimulationThreadPool
     */
    private SimulationThreadPool _simulationThreadPool;

    /**
     * This lock is used to synchronize access to all method calls that operate
     * on the simulation state. For usage details see {@link #getLock()}.
     * @deprecated Use of this field directly is deprecated, in future versions
     *             it will only be available by using the getter {@code getLock}.
     */
    @Deprecated
    public static final InheritableSimulationThreadLock lock =
        new InheritableSimulationThreadLock();

    /**
     * Version history:
     * 1 no header, included NetInstances and SearchQueue
     * 2 header(label, version, simulator type), NetInstances, Nets,
     *   SearchQueue - since Renew 1.2 beta 11
     * 3 small changes to ReflectionSerializer (Class[], nulls) can
     *   still read streams of version 2
     * 4 marking and search queue are now saved with time stamps
     *   incompatible change
     * 5 PlaceInstance is now an abstract class with subclasses
     *   incompatible change
     * (6) used in branch agent_serialization
     * 7 Different ID handling for net elements, introduction of remote
     *   layer - incompatible change
     * 8 Decomposed package de.renew.simulator - incompatible change
     * 9 Java version change, added assertions - incompatible change
     */
    private static final int STATE_STREAM_VERSION = 9;

    /**
     * The header identification string for all saved state streams.
     */
    private static final String STATE_STREAM_LABEL = "RenewState";

    /**
     * The name of the main package of this class.
     */
    private static final String MAIN_PACKAGE_NAME = "de.renew.simulator";

    /**
     * Name for the {@link SimulationControlCommand} when registering and removing
     * the command at {@link PluginManager#addCLCommand addCLCommand} and
     * {@link PluginManager#removeCLCommand removeCLCommand}
     */
    private static final String SIMULATION_CONTROL_COMMAND = "simulation";

    /**
     * Name for the {@link StartSimulationCommand} when registering and removing
     * the command at {@link PluginManager#addCLCommand addCLCommand} and
     * {@link PluginManager#removeCLCommand removeCLCommand}
     */
    private static final String SIMULATION_START_COMMAND = "startsimulation";

    /**
     * Creates an instance of the simulator component facade. There can exist at
     * most one instance at any time, additional constructor calls will throw an
     * exception. To get rid of the instance, call {@link #cleanup}. Any
     * subsequent calls to methods of the old instance will raise runtime
     * exceptions.
     *
     * @param props a <code>PluginProperties</code> value
     * @throws SingletonException if another singleton instance exists.
     */
    public SimulatorPlugin(PluginProperties props) {
        super(props);
        synchronized (SINGLETON_LOCK) {
            if (_singleton != null) {
                throw new SingletonException("At most one instance of SimulatorPlugin is allowed.");
            }
            _singleton = this;
        }
        _simulationThreadPool = SimulationThreadPool.getCurrent();
    }

    /**
     * This method <b>must</b> be called at the entry point of all public
     * methods of this object to ensure that this instance is still the
     * <code>SimulatorPlugin</code> singleton instance.
     *
     * @throws SingletonException if this instance is not the
     *         <code>SimulatorPlugin</code> singleton instance anymore.
     */
    private void checkSingleton() {
        if (_singleton != this) {
            throw new SingletonException();
        }
    }

    /**
     * This method should be called at the entry point of all public methods
     * that need an existing simulation setup.
     *
     * @throws NoSimulationException if no simulation is set up at the time being.
     */
    private void checkSimulation() throws NoSimulationException {
        if ((_currentSimulation == null) || (_currentSimulation.getSimulator() == null)) {
            throw new NoSimulationException();
        }
    }

    /**
     * This method is called when setting up a new simulation and ensures that
     * no other simulation is running.
     *
     * @throws SimulationRunningException if a simulation is already running.
     */
    private void checkNoSimulation() throws SimulationRunningException {
        if (_currentSimulation != null) {
            throw new SimulationRunningException();
        }
    }

    /**
     * Initialises this plugin after all dependencies to other plugins have been
     * fulfilled.
     *
     * @throws SingletonException if this instance is not the
     *         <code>SimulatorPlugin</code> singleton instance anymore.
     */
    @Override
    public void init() {
        checkSingleton();
        setDefaultNetLoader();
        // addExtension(new RemoteExtension());
        registerDefaultNetFinder(new SNSFinder());
        PluginManager.getInstance()
            .addCLCommand(SIMULATION_START_COMMAND, new StartSimulationCommand(this));
        PluginManager.getInstance()
            .addCLCommand(SIMULATION_CONTROL_COMMAND, new SimulationControlCommand(this));
    }

    /**
     * Returns the lock that is used to synchronize access to all method calls
     * that operate on the simulation state. Since all these methods include
     * convenience wrapping and locking, users of this plugin need not worry about
     * it unless they want to ensure atomic execution of multiple operations.
     * <p>
     * When using the lock it must be acquired <em>before</em> wrapping
     * execution within a simulation thread of the simulation thread pool. It is
     * also strongly recommended to include the unlock statement in a
     * try-finally block. Example:
     * </p>
     *
     * <pre><code>
     *   SimulatorPlugin.getLock().lock();
     *   try {
     *       Future future = SimulationThreadPool.getCurrent().submitAndWait(new Callable() {
     *           public Object call()
     *                  throws SomeException {
     *                                 ...
     *                  }
     *       });
     *
     *              return future.get();
     *   } catch (ExecutionException e) {
     *       ...
     *   } finally {
     *           SimulatorPlugin.getLock().unlock();
     *   }
     * </code></pre>
     */
    public static InheritableSimulationThreadLock getLock() {
        return lock;
    }

    @Override
    public void addExtension(SimulatorExtension ext) {
        checkSingleton();
        LOGGER.debug("SimulatorPlugin: Registering extension " + ext + ".");
        _extensions.add(ext);
    }

    @Override
    public void removeExtension(SimulatorExtension ext) {
        checkSingleton();
        LOGGER.debug("SimulatorPlugin: Deregistering extension " + ext + ".");
        _extensions.remove(ext);
    }

    @Override
    public void setupSimulation(final Properties props) {
        checkSingleton();

        lock.lock();

        try {
            SimulationThreadPool.getNew().executeAndWait(() -> {
                // Check that no simulation is running. The old behavior
                // of terminating a simulation (Renew 2.1) was in conflict
                // with the SimulationThreadPool (introduced Renew 2.2).
                // Terminating a simulation is an asynchronous process
                // in contrast to the immediate replacement of a current
                // thread pool during setup. It would be probable that
                // some pending events of the old simulation would be
                // executed within the new thread pool.
                try {
                    checkNoSimulation();
                } catch (SimulationRunningException e) {
                    SimulationThreadPool.discardNew();
                    throw e;
                }

                restartThreadPool();
                assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";

                // Combine the plugin properties with the specified
                // properties from this method call. Disconnect the
                // active property set from the plugin properties by
                // copying all entries.
                final Properties activeProperties = new Properties();
                activeProperties.putAll(getProperties());
                if (props != null) {
                    activeProperties.putAll(props);
                }
                int maxPriority = PropertyHelper
                    .getIntProperty(activeProperties, PRIORITY_PROP_NAME, Thread.NORM_PRIORITY);
                if (_simulationThreadPool.getMaxPriority() != maxPriority) {
                    _simulationThreadPool.setMaxPriority(maxPriority);
                }
                SimulatorEventQueue.initialize();
                LOGGER.debug("SimulatorPlugin: Setting up simulation.");
                // Take a snapshot of registered extensions, these are
                // now our active extensions.
                SimulatorExtension[] activeExtensions =
                    _extensions.toArray(new SimulatorExtension[0]);
                // Configure class reloading, if requested.
                possiblySetupClassSource(activeProperties);

                // Ensure that all old nets have been forgotten and
                // set the new net loader.
                Net.forgetAllNets();
                Net.setNetLoader(_nextNetLoader);
                if (_nextNetLoader instanceof DelayedDelegationNetLoader) {
                    synchronized (_registeredFinders) {
                        LOGGER.debug("SimulatorPlugin: Creating default shadow net loader.");
                        _currentShadowNetLoader = new DefaultShadowNetLoader(activeProperties);
                        for (Finder finder : _registeredFinders) {
                            _currentShadowNetLoader.registerFinder(finder);
                        }
                        for (PathlessFinder finder : _registeredPathlessFinders) {
                            _currentShadowNetLoader.registerPathlessFinder(finder);
                        }
                    }
                    LOGGER.debug("SimulatorPlugin: Configuring delayed net loader.");
                    ((DelayedDelegationNetLoader) _nextNetLoader)
                        .setNetLoader(new DefaultCompiledNetLoader(_currentShadowNetLoader));
                }

                // Create the simulation engine with respect to the
                // current properties.
                Simulator simulator = newSimulator(activeProperties);

                // Store all information in a new simulation environment.
                _currentSimulation =
                    new SimulationEnvironment(simulator, activeExtensions, activeProperties);
                _virginSimulation = true;

                // Inform all active extensions about the simulation setup.
                for (SimulatorExtension activeExtension : activeExtensions) {
                    activeExtension.simulationSetup(_currentSimulation);
                }

                // Register this as exit blocker as long as the simulation
                // is
                // active.
                PluginManager.getInstance().blockExit(SimulatorPlugin.getCurrent());
            });
        } finally {
            lock.unlock();
        }
    }

    /**
     * Create a new simulator. The actual object that is created depends on the
     * given properties. The properties will be updated with respect to the
     * chosen simulator class and multiplicity. A zero multiplicity denotes a
     * sequential simulator, which is minimally concurrent.
     * <p>
     * This method must be called from a simulation thread.
     */
    private static Simulator newSimulator(final Properties props) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        final int simulatorMode = PropertyHelper.getIntProperty(props, MODE_PROP_NAME, 1);
        int simulatorMultiplicity = PropertyHelper.getIntProperty(props, MULTIPLICITY_PROP_NAME, 1);

        Class<?> simulatorClass =
            PropertyHelper.getClassProperty(props, CLASS_PROP_NAME, Simulator.class);

        // Is the simulator class already set?
        if (simulatorClass != null) {
            LOGGER.info(
                "Using simulator class " + simulatorClass.getName() + " with "
                    + simulatorMultiplicity + " simulators ...");
        } else {
            // The simulator class must be set from the simulator mode.

            // What is the simulator mode?
            if (simulatorMode == 1) {
                simulatorClass = ConcurrentSimulator.class;
                LOGGER.info("Using default concurrent simulator ...");
            } else if (simulatorMode == 0 || simulatorMode == -1) {
                simulatorClass = NonConcurrentSimulator.class;
                LOGGER.info("Using sequential simulator ...");
            } else {
                // Use multiple simulators.
                simulatorClass = ParallelSimulator.class;
                // Set the number of simulators to use.
                simulatorMultiplicity = simulatorMode;

                if (simulatorMultiplicity < 0) {
                    LOGGER.warn("Using " + (-simulatorMultiplicity) + " sequential simulators ...");
                } else {
                    LOGGER.warn("Using " + simulatorMultiplicity + " concurrent simulators ...");
                }
                LOGGER.warn("Caution! This is an experimental feature!");
            }
        }

        final boolean eagerSimulation = PropertyHelper.getBoolProperty(props, EAGER_PROP_NAME);
        if (eagerSimulation) {
            LOGGER.info("Using eager simulation mode.");
        }

        // Create instance of given class with java.lang.reflect.
        // This code was inspired by de.renew.remote.RemoteExtension.createServer(Class<?>).
        Simulator simulator;
        try {
            try {
                // Get constructor taking one integer and one boolean as parameters
                // and apply it with simulatorMultiplicity and !eagerSimulation.
                // See {@link Simulator} documentation.
                simulator = (Simulator) simulatorClass.getConstructor(int.class, boolean.class)
                    .newInstance(simulatorMultiplicity, !eagerSimulation);
            } catch (NoSuchMethodException e) {
                // If the needed constructor does not exist try the following instead:
                // Get constructor taking one boolean as parameter and apply it with !eagerSimulation.
                // See {@link Simulator} documentation.
                simulator = (Simulator) simulatorClass.getConstructor(boolean.class)
                    .newInstance(!eagerSimulation);
                // Set multiplicity to 0 if its sequential and 1 otherwise.
                simulatorMultiplicity = (simulator.isSequential() ? 0 : 1);
            }

            // Catch all declared exceptions.
        } catch (SecurityException | NoSuchMethodException | IllegalArgumentException
            | InstantiationException | IllegalAccessException | InvocationTargetException e) {
            LOGGER.error("Encountered exception when trying to instantiate simulator class:", e);
            simulator = new ConcurrentSimulator(!eagerSimulation);
            simulatorClass = ConcurrentSimulator.class;
            simulatorMultiplicity = 1;
        }

        props.setProperty(CLASS_PROP_NAME, simulatorClass.getName());
        props.setProperty(MULTIPLICITY_PROP_NAME, Integer.toString(simulatorMultiplicity));

        return simulator;
    }

    @Override
    public boolean isSimulationActive() {
        checkSingleton();

        return _currentSimulation != null && _currentSimulation.getSimulator().isActive();
    }

    @Override
    public SimulationEnvironment getCurrentEnvironment() {
        return _currentSimulation;
    }

    @Override
    public ShadowLookup insertNets(final ShadowNetSystem netSystem)
        throws SyntaxException, NoSimulationException
    {
        checkSingleton();

        lock.lock();
        try {
            Future<ShadowLookup> future = SimulationThreadPool.getCurrent().submitAndWait(() -> {
                ShadowLookup lookup;
                checkSimulation();

                Objects.requireNonNull(netSystem, "Missing shadow net system.");

                // Apply default net loader if requested.
                ShadowNetLoader netLoader = netSystem.getNetLoader();
                if ((netLoader == null) && (_currentShadowNetLoader != null)) {
                    LOGGER.debug(
                        "SimulatorPlugin: Applying default shadow net loader to net system.");
                    netSystem.setNetLoader(_currentShadowNetLoader);
                }

                // Compile nets.
                if (_virginSimulation) {
                    LOGGER.debug("SimulatorPlugin: Compiling first net system.");
                    lookup = netSystem.compile();
                } else {
                    LOGGER.debug("SimulatorPlugin: Adding another net system.");
                    lookup = netSystem.compileMore();
                }
                LOGGER.debug("SimulatorPlugin: Compilation result lookup: " + lookup);

                // Check whether compilation result fits into
                // the
                // current
                // simulation with respect to sequential step
                // requirements.
                SequentialOnlyExtension seqEx = SequentialOnlyExtension.lookup(lookup);
                boolean sequentialOnly = seqEx.getSequentialOnly();
                if (sequentialOnly && !_currentSimulation.getSimulator().isSequential()) {
                    throw new SyntaxException("Some nets need a sequential simulator.");
                    // TODO: add error objects by asking seqEx
                }

                // Now we are sure that the nets can be added to
                // the
                // simulation.
                // Unset the virgin flag.
                _virginSimulation = false;

                // Inform all active extensions about the new
                // nets.
                SimulatorExtension[] activeExtensions = _currentSimulation.getExtensions();
                for (SimulatorExtension activeExtension : activeExtensions) {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug(
                            SimulatorPlugin.class.getName() + ": Active Extension compile net "
                                + activeExtension.toString());
                    }
                    activeExtension.netsCompiled(lookup);
                }

                // Insert all compiled nets into the running
                // simulation.
                lookup.makeNetsKnown();
                return lookup;
            });

            return future.get();
        } catch (InterruptedException e) {
            LOGGER.info("Simulation ended");
        } catch (ExecutionException e) {
            Throwable t = e.getCause();
            if (t instanceof SyntaxException exc) {
                throw exc;
            } else if (t instanceof NoSimulationException exc) {
                throw exc;
            } else if (t instanceof RuntimeException exc) {
                throw exc;
            } else if (t instanceof Error exc) {
                throw exc;
            } else {
                LOGGER.error("Simulation thread threw an exception", e);
            }
        } finally {
            lock.unlock();
        }

        // We should never return nothing but some error occurred before.
        return null;

    }

    @Override
    public SimulationState restoreStateFromDatabase() {
        checkSingleton();

        lock.lock();
        try {
            Future<SimulationState> future = SimulationThreadPool.getCurrent().submitAndWait(() -> {
                checkSimulation();
                return SetupHelper.setup(_currentSimulation.getProperties());
            });
            return future.get();
        } catch (InterruptedException e) {
            LOGGER.error("Timeout while waiting for simulation thread to finish", e);
        } catch (ExecutionException e) {
            Throwable t = e.getCause();
            if (t instanceof NoSimulationException exc) {
                throw exc;
            } else if (t instanceof RuntimeException exc) {
                throw exc;
            } else if (t instanceof Error exc) {
                throw exc;
            } else {
                LOGGER.error("Simulation thread threw an exception", e);
            }
        } finally {
            lock.unlock();
        }

        // We should never return nothing but some error occurred before.
        return null;
    }

    @Override
    public void saveState(final ObjectOutput output, final NetInstance[] instances)
        throws IOException
    {
        checkSingleton();

        lock.lock();
        try {
            Future<Object> future = _simulationThreadPool.submitAndWait(() -> {
                checkSimulation();
                _currentSimulation.getSimulator().stopRun();

                // Use the domain trace feature of the
                // RenewObjectOutputStream, if available.
                RenewObjectOutputStream rOut = null;
                if (output instanceof RenewObjectOutputStream) {
                    rOut = (RenewObjectOutputStream) output;

                }
                if (rOut != null) {
                    rOut.beginDomain(SimulatorPlugin.class);
                }

                // Write the header, which contains:
                // - label
                // - file format version number
                // - type of simulator used
                output.writeObject(STATE_STREAM_LABEL);
                output.writeInt(STATE_STREAM_VERSION);
                output.writeInt(
                    PropertyHelper.getIntProperty(
                        _currentSimulation.getProperties(), MULTIPLICITY_PROP_NAME));

                // First part: save all NetInstances explicitly
                // named.
                // They are not necessarily sufficient to describe
                // the
                // simulation state completely.
                output.writeInt(instances.length);
                for (NetInstance instance : instances) {
                    output.writeObject(instance);
                }

                // If a RenewObjectOutputStream is used, write
                // all delayed fields NOW.
                if (rOut != null) {
                    rOut.writeDelayedObjects();

                }

                // Second part: save all Nets currently known by
                // the Net.forName() lookup mechanism.
                // This ensures that the static part of all compiled
                // nets will be available on deserialization.
                Net.saveAllNets(output);

                // Third part: add all entries from the SearchQueue.
                // These entries alone are sufficient to describe
                // the
                // current simulation state completely.
                // But this information does not necessarily
                // include
                // all
                // nets possibly required by future simulation
                // steps.
                SearchQueue.saveQueue(output);

                // Last part: Restore the ID registry with its
                // well-known
                // IDs for every token.
                IDRegistry.save(output);

                if (rOut != null) {
                    rOut.endDomain(SimulatorPlugin.class);
                }
                return null;
            });
            future.get();
        } catch (InterruptedException e) {
            LOGGER.error("Timeout while waiting for simulation thread to finish", e);
        } catch (ExecutionException e) {
            Throwable t = e.getCause();
            if (t instanceof IOException exc) {
                throw exc;
            } else if (t instanceof NoSimulationException exc) {
                throw exc;
            } else if (t instanceof RuntimeException exc) {
                throw exc;
            } else if (t instanceof Error exc) {
                throw exc;
            } else {
                LOGGER.error("Simulation thread threw an exception", e);
            }
        } finally {
            lock.unlock();
        }
    }

    @Override
    public NetInstance[] loadState(final ObjectInput input, final Properties props)
        throws IOException, ClassNotFoundException, SimulationRunningException
    {
        checkSingleton();

        lock.lock();
        try {
            Future<NetInstance[]> future = SimulationThreadPool.getNew().submitAndWait(() -> {
                List<NetInstance> explicitInstances;
                try {
                    LOGGER.debug("Loading simulation state...");

                    setupSimulation(props);

                    // Check for valid header, which includes:
                    // - label
                    // - file format version number
                    // - type of simulator used
                    String streamLabel = (String) input.readObject();

                    if (!streamLabel.equals(STATE_STREAM_LABEL)) {
                        throw new StreamCorruptedException(
                            "Stream does not seem to contain renew state data.");
                    }
                    int streamVersion = input.readInt();

                    // that's most probably not ok.
                    if (streamVersion != STATE_STREAM_VERSION) {
                        throw new StreamCorruptedException(
                            "State data is of different version " + "(" + streamVersion
                                + ") than the current version (" + STATE_STREAM_VERSION + ").");
                    }

                    int simulatorMultiplicity = PropertyHelper.getIntProperty(
                        _currentSimulation.getProperties(), MULTIPLICITY_PROP_NAME, 1);
                    int streamSimulatorMultiplicity = input.readInt();

                    if (streamSimulatorMultiplicity != simulatorMultiplicity) {
                        LOGGER.warn(
                            "Simulation state was saved "
                                + "using a different simulator multiplicity " + "("
                                + streamSimulatorMultiplicity + ") " + "than currently selected ("
                                + simulatorMultiplicity + ").");
                    }

                    // First part: read all NetInstances stored
                    // explicitly.
                    int count = input.readInt();
                    explicitInstances = new ArrayList<>(count);
                    try {
                        for (int i = 0; i < count; i++) {
                            explicitInstances
                                .add((NetInstance) de.renew.util.ClassSource.readObject(input));
                        }
                    } catch (ClassCastException e) {
                        LOGGER.debug(e.getMessage(), e);
                        throw new StreamCorruptedException(
                            "Object other than NetInstance found "
                                + "when looking for net instances: " + e.getMessage());
                    }

                    // If a RenewObjectInputStream is used, read
                    // all delayed fields NOW.
                    if (input instanceof RenewObjectInputStream) {
                        ((RenewObjectInputStream) input).readDelayedObjects();
                    }

                    // Second part: read all compiled Nets
                    Net.loadNets(input);

                    // Third part: add all necessary entries to the
                    // SearchQueue.
                    SearchQueue.loadQueue(input);

                    // Last part: reestablish the global ID registry
                    // for
                    // token IDs.
                    IDRegistry.load(input);
                } catch (IOException e) {
                    // If an exception occurs, pass it to the
                    // caller.
                    // But restore a known state first - the 'no
                    // simulation active' state.
                    terminateSimulation();
                    throw e;
                } catch (ClassNotFoundException | StackOverflowError e) {
                    // The same as above, but for another exception
                    // type...
                    terminateSimulation();
                    throw e;
                }

                // Return the list of read NetInstances.
                return explicitInstances.toArray(new NetInstance[0]);
            });
            return future.get();
        } catch (InterruptedException e) {
            LOGGER.error("Timeout while waiting for simulation thread to finish", e);
        } catch (ExecutionException e) {
            Throwable t = e.getCause();
            if (t instanceof SimulationRunningException) {
                throw new SimulationRunningException(e);
            } else if (t instanceof IOException exc) {
                throw exc;
            } else if (t instanceof ClassNotFoundException) {
                throw new ClassNotFoundException(t.getMessage(), e);
            } else if (t instanceof RuntimeException exc) {
                throw exc;
            } else if (t instanceof Error exc) {
                throw exc;
            } else {
                LOGGER.error("Simulation thread threw an exception", e);
            }
        } finally {
            lock.unlock();
        }

        // We should never return nothing but some error occurred before.
        return null;
    }

    @Override
    public synchronized void setNetLoader(NetLoader loader) {
        checkSingleton();
        LOGGER.debug("SimulatorPlugin: Configuring net loader " + loader + ".");
        _nextNetLoader = loader;
    }

    @Override
    public synchronized void setDefaultNetLoader() {
        setNetLoader(new DelayedDelegationNetLoader());
    }

    @Override
    public void registerDefaultNetFinder(final Finder finder) {
        synchronized (_registeredFinders) {
            _registeredFinders.add(finder);
            if (_currentShadowNetLoader != null) {
                _currentShadowNetLoader.registerFinder(finder);
            }
        }
    }

    @Override
    public void removeDefaultNetFinder(final Finder finder) {
        synchronized (_registeredFinders) {
            _registeredFinders.remove(finder);
            if (_currentShadowNetLoader != null) {
                _currentShadowNetLoader.removeFinder(finder);
            }
        }
    }

    @Override
    public void registerDefaultPathlessFinder(final PathlessFinder finder) {
        synchronized (_registeredFinders) {
            _registeredPathlessFinders.add(finder);
            if (_currentShadowNetLoader != null) {
                _currentShadowNetLoader.registerPathlessFinder(finder);
            }
        }
    }

    @Override
    public void removeDefaultPathlessFinder(final PathlessFinder finder) {
        synchronized (_registeredFinders) {
            _registeredPathlessFinders.remove(finder);
            if (_currentShadowNetLoader != null) {
                _currentShadowNetLoader.removePathlessFinder(finder);
            }
        }
    }

    @Override
    public NetInstance createNetInstance(final String net)
        throws NetNotFoundException, NoSimulationException
    {
        checkSingleton();

        lock.lock();
        try {
            Future<NetInstance> future = SimulationThreadPool.getCurrent().submitAndWait(() -> {
                NetInstance netInstance = null;
                checkSimulation();

                // Create the first net instance
                if (net != null) {
                    Net netTemplate = Net.forName(net);
                    if (netTemplate != null) {
                        netInstance = netTemplate
                            .buildInstance(_currentSimulation.getSimulator().nextStepIdentifier());
                        _currentSimulation.getSimulator().refresh();
                    }
                }
                return netInstance;
            });

            return future.get();
        } catch (InterruptedException e) {
            LOGGER.info("Creation of NetInstances was aborted");
        } catch (ExecutionException e) {
            Throwable t = e.getCause();
            if (t instanceof NetNotFoundException exc) {
                throw exc;
            } else if (t instanceof NoSimulationException exc) {
                throw exc;
            } else if (t instanceof RuntimeException exc) {
                throw exc;
            } else if (t instanceof Error exc) {
                throw exc;
            } else {
                LOGGER.error("Simulation thread threw an exception", e);
            }
        } finally {
            lock.unlock();
        }

        // We should never return nothing but some error occurred before.
        return null;

    }

    /**
     * Terminates the current simulation. If no simulation has been set up,
     * nothing happens.
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link #lock lock}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     * <p>
     * {@link #lock}
     */
    public void terminateSimulation() {
        checkSingleton();

        lock.lock();

        try {
            SimulationThreadPool.getCurrent().executeAndWait(() -> {
                if (_currentSimulation != null) {
                    LOGGER.debug("SimulatorPlugin: Stopping simulation.");

                    SimulatorExtension[] exts = _currentSimulation.getExtensions();
                    for (SimulatorExtension ext : exts) {
                        ext.simulationTerminating();
                    }

                    // Stop the engine.
                    _currentSimulation.getSimulator().terminateRun();

                    exts = _currentSimulation.getExtensions();
                    for (SimulatorExtension ext : exts) {
                        ext.simulationTerminated();
                    }

                    // Go back to simulation time 0 and
                    // clear all outstanding search requests.
                    // SearchQueue.reset(0);
                    SearchQueue.reset(0);
                    // Clear all token IDs that might be
                    // still registered.
                    IDRegistry.reset();

                    // Forget all net structures.
                    Net.forgetAllNets();

                    // Retract our exit blocker because the simulation is
                    // over.
                    PluginManager.getInstance().exitOk(SimulatorPlugin.getCurrent());

                    // We should not clean the thread pool here.
                    // Simulation is terminated asynchronously.
                    // There still may be requests for a simulation thread.
                    // SimulationThreadPool.getCurrent().cleanup();
                    _currentSimulation = null;
                    _currentShadowNetLoader = null;
                }
            });
        } finally {
            lock.unlock();
        }
    }

    private void restartThreadPool() {
        SimulationThreadPool.cleanup();
        _simulationThreadPool = SimulationThreadPool.getSimulationThreadPool();
    }

    /**
     * Stops any running simulation and clears all data used by the simulator.
     * If the cleanup was successful (returns <code>true</code>) this
     * <code>SimulatorPlugin</code> instance is rendered useless and all future
     * method calls will throw <code>SingletonException</code>s.
     *
     * @return <code>true</code>, if the cleanup was successfully finished and
     *         this object has lost its singleton status. <br>
     *         <code>false</code>, if the cleanup failed. This object is still
     *         the singleton for any simulator access.
     * @throws SingletonException if this object has already lost the singleton instance status
     *         before.
     */
    @Override
    public synchronized boolean cleanup() {
        setNetLoader(null);

        synchronized (SINGLETON_LOCK) {
            checkSingleton();
            terminateSimulation();

            // if (result=false) return immediately before releasing
            // the singleton!
            _singleton = null;
        }
        PluginManager.getInstance().removeCLCommand(SIMULATION_START_COMMAND);
        PluginManager.getInstance().removeCLCommand(SIMULATION_CONTROL_COMMAND);

        SimulationThreadPool.cleanup();
        _simulationThreadPool = null;

        return true;
    }

    /**
     * Describe <code>canShutDown</code> method here.
     *
     * @return a <code>boolean</code> value
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     */
    @Override
    public boolean canShutDown() {
        checkSingleton();
        return true;
    }

    /**
     * Executes the given {@link Callable} in a simulation thread and waits for
     * its computation to finish.
     * <p>
     * This facade method just delegates to the current
     * {@link SimulationThreadPool} instance.  For the caller's convenience, the
     * resulting {@link Future} is unpacked immediately.  If an
     * {@link ExecutionException} occurs, the causing exception is unpacked or
     * converted into a {@link RuntimeException}.  If there is no simulation
     * running, this method exits with a {@link NoSimulationException}.
     * </p>
     *
     * @param <T> the type of the computation
     * @param task the computation to be executed in a simulation thread.
     * @return the computation result.
     * @throws SingletonException if this object is not the simulator
     *         plugin singleton instance anymore.
     * @throws NoSimulationException if there is no simulation set up.
     * @throws InterruptedException if the current thread is interrupted
     * @see SimulationThreadPool#submitAndWait(Callable)
     **/
    public <T> T submitAndWait(Callable<T> task) throws InterruptedException {
        checkSingleton();
        lock.lock();
        try {
            checkSimulation();
            Future<T> futureObj = _simulationThreadPool.submitAndWait(task);
            return futureObj.get();
        } catch (ExecutionException e) {
            Throwable t = e.getCause();
            if (t instanceof NoSimulationException exc) {
                throw exc;
            } else if (t instanceof RuntimeException exc) {
                throw exc;
            } else if (t instanceof Error exc) {
                throw exc;
            } else {
                LOGGER.error("Simulation thread threw an exception: " + t, e);
                throw new RuntimeException(e);
            }
        } finally {
            SimulatorPlugin.lock.unlock();
        }
    }

    @Override
    public void possiblySetupClassSource(final Properties props) throws IllegalStateException {
        checkSingleton();

        lock.lock();
        try {
            Future<Object> result = _simulationThreadPool.submitAndWait(() -> {
                if (isSimulationActive()) {
                    throw new IllegalStateException(
                        "Reconfiguration of class source is "
                            + "not allowed while a simulation is running.");
                }
                boolean classReinit = PropertyHelper.getBoolProperty(props, REINIT_PROP_NAME);
                if (classReinit) {
                    if (!_previousClassReinit) {
                        LOGGER.info("Using classReinit mode.");
                    } else {
                        LOGGER.debug("SimulatorPlugin: Re-initialising class loader.");
                    }

                    // In Renew 2.x SelectiveClassLoader was
                    // replaced by
                    // BottomClassLoader.
                    //
                    // SelectiveClassLoader classLoader = new
                    // SelectiveClassLoader();
                    // classLoader.setSelectors(new String[] {
                    // "de.renew.util.ReloadableDeserializerImpl",
                    // "-java",
                    // "-collections.", "-CH.ifa.draw.",
                    // "-de.renew.",
                    // "-de.uni_hamburg.fs." });
                    ClassLoader classLoader = PluginManager.getInstance().getNewBottomClassLoader();
                    ClassSource.setClassLoader(classLoader);
                } else if (_previousClassReinit) {
                    LOGGER.debug("classReinit mode disabled.");
                    ClassSource.setClassLoader(null);
                }
                _previousClassReinit = classReinit;

                return null;
            });

            result.get();
        } catch (InterruptedException e) {
            LOGGER.error("Timeout while waiting for simulation thread to finish", e);
        } catch (ExecutionException e) {
            Throwable t = e.getCause();
            if (t instanceof IllegalStateException exc) {
                throw exc;
            } else if (t instanceof RuntimeException exc) {
                throw exc;
            } else if (t instanceof Error exc) {
                throw exc;
            } else {
                LOGGER.error("Simulation thread threw an exception", e);
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * Provides a reference to the current Renew Simulator plugin instance. The
     * instance is queried from the plugin management system. So the result will
     * be <code>null</code>, if the simulator plugin is not activated.
     *
     * @return the active simulator plugin instance, if there is any. Returns
     *         <code>null</code> otherwise.
     */
    public static SimulatorPlugin getCurrent() {
        for (IPlugin plugin : PluginManager.getInstance().getPluginsProviding(MAIN_PACKAGE_NAME)) {
            if (plugin instanceof SimulatorPlugin) {
                return (SimulatorPlugin) plugin;
            }
        }
        return _singleton;
    }

    /**
     * This net loader serves as a placeholder. It denies all
     * <code>loadNet()</code> requests until the real net loader has been
     * configured.
     */
    private static class DelayedDelegationNetLoader implements NetLoader {
        private NetLoader _netLoader = null;

        public void setNetLoader(NetLoader netLoader) {
            _netLoader = netLoader;
        }

        @Override
        public Net loadNet(final String netName) throws NetNotFoundException {
            if (_netLoader == null) {
                throw new NetNotFoundException("No net loader configured.");
            } else {
                return _netLoader.loadNet(netName);
            }
        }
    }
}