package de.renew.net.arc;

import java.util.Collection;
import java.util.Vector;

import de.renew.engine.searcher.Occurrence;
import de.renew.engine.searcher.Searcher;
import de.renew.expression.Expression;
import de.renew.expression.VariableMapper;
import de.renew.net.NetInstance;
import de.renew.net.Place;
import de.renew.net.Transition;
import de.renew.net.TransitionInscription;


/**
 * An {@code Arc} instance represents an arc from or to a place.
 * <p>
 * Arcs can be of different types distinguished by the public
 * class constants defined in this class.
 * The arc types that are annotated with an asterisk require a
 * time expression.
 * </p>
 **/
public class Arc implements TransitionInscription {

    /** Arc type: input arc (*). **/
    public static final int in = -1;

    /** Arc type: test arc. **/
    public static final int test = 0;

    /** Arc type: output arc (*). **/
    public static final int out = 1;

    /**
     * Arc type: double arc (*) that reserves the token and
     * applies the time to the input token.
     **/
    public static final int both = 2;

    /**
     * Arc type: double arc (*) that releases the token early
     * during the firing.
     **/
    public static final int fastBoth = 3;

    /**
     * Arc type: test arc that releases the token early during
     * the firing.
     **/
    public static final int fastTest = 4;

    /** Arc type: inhibitor arc. **/
    public static final int inhibitor = 5;

    /**
     * Arc type: double arc (*) that reserves the token and
     * applies the time to the output token.
     **/
    public static final int bothOT = 6;

    /**
     * The place that the arc connects to.
     */
    protected Place place;
    /**
     * The transition that the arc connects to.
     */
    protected Transition transition;
    /**
     * The arc's type.
     */
    protected int arcType;
    /**
     * The arc's token expression.
     */
    protected Expression tokenExpr;
    /**
     * The arc's time expression.
     */
    protected Expression timeExpr;
    /**
     * {@code true}, if tracing should be activated or {@code false}, otherwise.
     */
    boolean trace;

    /**
     * The constructor for Arc.
     *
     * @param place the place the arc connects to
     * @param transition the transition the arc connects to
     * @param arcType the type of the arc
     * @param tokenExpr the token expression of the arc
     * @param timeExpr the time expression of the arc
     */
    public Arc(
        Place place, Transition transition, int arcType, Expression tokenExpr, Expression timeExpr)
    {
        if (arcType < in || arcType > bothOT) {
            throw new RuntimeException("Bad arc type: " + arcType + ".");
        }
        this.place = place;
        this.transition = transition;
        this.arcType = arcType;
        this.tokenExpr = tokenExpr;
        this.timeExpr = timeExpr;
        trace = true;
    }

    /**
     * Activate/Deactivate tracing on this arc according to the given parameter.
     *
     * @param trace {@code true}, if tracing is enabled or {@code false}, otherwise.
     */
    public void setTrace(boolean trace) {
        this.trace = trace;
    }

    /**
     * Returns {@code true} if tracing is enabled or {@code false} otherwise.
     *
     * @return {@code true}, if tracing is enabled or {@code false}, otherwise.
     */
    public boolean getTrace() {
        return trace;
    }

    /**
     * Returns {@code true} if the arc is a test arc or {@code false} otherwise.
     *
     * @return {@code true}, if the arc is a test arc or {@code false}, otherwise.
     */
    public boolean isTestArc() {
        return arcType == test || arcType == fastTest;
    }

    /**
     * Returns {@code true} if the arc is an untimed arc or {@code false} otherwise.
     *
     * @return {@code true}, if the arc is an untimed arc or {@code false}, otherwise.
     */
    public boolean isUntimedArc() {
        return isTestArc() || arcType == inhibitor;
    }

    @Override
    public java.util.Collection<Occurrence> makeOccurrences(
        VariableMapper mapper, NetInstance netInstance, Searcher searcher)
    {
        Collection<Occurrence> coll = new Vector<Occurrence>();
        coll.add(new ArcOccurrence(this, mapper, netInstance));
        return coll;
    }
}