package de.renew.net.arc;

import java.util.Collection;

import de.renew.engine.common.AssignBinder;
import de.renew.engine.searcher.TriggerableCollection;
import de.renew.net.SimulatablePlaceInstance;
import de.renew.unify.Variable;

/**
 * An {@code ArcAssignBinder} is used to find tokens that belong to a {@link SimulatablePlaceInstance} and match a
 * {@code Variable} instance's value. When binding, a valid token is nondeterministically selected and unified with
 * the {@code Variable}. It is used by an {@link ArcOccurrence} to find and bind tokens from the connecting
 * {@code SimulatablePlaceInstance}, using the arc's evaluated {@link de.renew.expression.Expression} as
 * {@code Variable}. This is only advisable if the arc's expression is invertible.
 */
class ArcAssignBinder extends AssignBinder {
    /**
     * The place instance which the {@code ArcAssignBinder} checks for valid tokens.
     */
    private SimulatablePlaceInstance placeInstance;
    /**
     * {@code true}, if the corresponding {@code Arc} is a test arc or {@code false}, otherwise.
     */
    private boolean wantTest;

    /**
     * Creates a new {@code ArcAssignBinder} using the given values.
     *
     * @param variable the {@code Variable} whose value will be used to find matching tokens and that will be unified
     *                 with a matching token when {@link AssignBinder#bind} is called
     * @param placeInstance the place instance connected to the arc
     * @param wantTest {@code true}, if the arc is a test arc or {@code false}, otherwise
     */
    public ArcAssignBinder(
        Variable variable, SimulatablePlaceInstance placeInstance, boolean wantTest)
    {
        super(variable, false);
        this.placeInstance = placeInstance;
        this.wantTest = wantTest;
    }

    /**
     * Locks the {@link de.renew.net.PlaceInstance#lock PlaceInstance} associated with this arc.
     **/
    @Override
    public final void lock() {
        placeInstance.lock.lock();
    }

    /**
     * Unlocks the {@link de.renew.net.PlaceInstance#lock PlaceInstance} associated with this arc.
     **/
    @Override
    public final void unlock() {
        placeInstance.lock.unlock();
    }

    @Override
    public Collection<Object> getCandidates(Object pattern) {
        if (wantTest) {
            return placeInstance.getDistinctTestableTokens(pattern);
        } else {
            return placeInstance.getDistinctTokens(pattern);
        }
    }

    @Override
    public TriggerableCollection getTriggerables() {
        return placeInstance.triggerables();
    }
}