package de.renew.net.arc;

import java.lang.reflect.Array;
import java.util.Vector;

import de.renew.engine.common.SimulatorEventLogger;
import de.renew.engine.common.StepIdentifier;
import de.renew.engine.events.Clearing;
import de.renew.engine.searcher.EarlyExecutable;
import de.renew.net.SimulatablePlaceInstance;
import de.renew.net.TransitionInstance;
import de.renew.unify.Impossible;
import de.renew.unify.Unify;
import de.renew.unify.Variable;
import de.renew.util.Value;

/**
 * The executable belonging to a {@link ClearArcOccurrence}.
 * It is used to clear all tokens from a {@link SimulatablePlaceInstance}.
 */
class ClearArcExecutable implements EarlyExecutable {
    /**
     * The place instance whose tokens are cleared.
     */
    SimulatablePlaceInstance pInstance;
    /**
     * The transition instance whose firing triggers the clearing of the place instance.
     */
    TransitionInstance tInstance;
    /**
     * The variable which will be bound to an array containing the tokens removed from the place instance.
     */
    Variable variable;
    /**
     * The clear arc whose occurrence leads to this {@code ClearArcExecutable} being executed.
     */
    ClearArc arc;
    /**
     * The tokens that the clear arc has removed from the place instance.
     */
    Vector<Object> removedTokens;
    /**
     * The time stamps of the times at which each of the removed tokens were removed from the place instance.
     */
    Vector<Double> removedTimeStamps;

    /**
     * Constructs a new {@code ClearArcExecutable} based on its place and transition instances, a {@code Variable} and
     * the {@code ClearArc} it is based on.
     * 
     * @param pInstance the place instance whose tokens will be cleared
     * @param tInstance the transition instance whose firing triggers the {@code ClearArcExecutable}
     * @param variable the variable in which the removed tokens will be stored
     * @param arc the {@code ClearArc} that the {@code ClearArcExecutable} is based on
     */
    ClearArcExecutable(
        SimulatablePlaceInstance pInstance, TransitionInstance tInstance, Variable variable,
        ClearArc arc)
    {
        this.pInstance = pInstance;
        this.tInstance = tInstance;
        this.variable = variable;
        this.arc = arc;
    }

    @Override
    public long lockPriority() {
        return pInstance.lockOrder;
    }

    @Override
    public int phase() {
        return CLEAR;
    }

    /**
     * Locks the {@link de.renew.net.PlaceInstance#lock PlaceInstance} associated with this arc.
     **/
    @Override
    public void lock() {
        pInstance.lock.lock();
    }

    @Override
    public void verify(StepIdentifier stepIdentifier) throws Impossible {
        // Ensure large size increments.
        removedTokens = new Vector<Object>(8, 0);
        removedTimeStamps = new Vector<Double>(8, 0);

        pInstance.extractAllTokens(removedTokens, removedTimeStamps);

        Object result = Array.newInstance(arc.elementType, removedTokens.size());
        for (int i = 0; i < removedTokens.size(); i++) {
            Object value = removedTokens.elementAt(i);
            if (arc.elementType.isPrimitive()) {
                value = Value.unvalueAndCast(value, arc.elementType);
            }
            Array.set(result, i, value);
        }

        Unify.unify(variable, result, null);
    }

    @Override
    public void execute(StepIdentifier stepIdentifier) {
        if (arc.getTrace()) {
            // log activities on net level
            SimulatorEventLogger.log(stepIdentifier, new Clearing(pInstance), pInstance);
        }
    }

    @Override
    public void rollback() {
        // We have to undo the previous removal. We cannot do this
        // without notifying the observers and listeners, because
        // it was not done silently. However, the database must not log
        // this modification.
        for (int i = 0; i < removedTokens.size(); i++) {
            Object token = removedTokens.elementAt(i);
            double time = removedTimeStamps.elementAt(i).doubleValue();
            pInstance.internallyInsertToken(token, time, false);
        }
        removedTokens = null;
    }

    /**
     * Unlocks the {@link de.renew.net.PlaceInstance#lock PlaceInstance} associated with this arc.
     **/
    @Override
    public void unlock() {
        pInstance.lock.unlock();
    }
}