package de.renew.unify;

import java.util.HashSet;
import java.util.Set;

/**
 * This class checks if a calculation can actually be performed.
 * <p>
 * On the fly:
 *   Check that no calculated unknown is bound.
 *   Check that no calculated unknown is registered twice.
 *   Check that no two calculated unknowns depend on each other.
 * During the consistency check:
 *   Check that all early variables have become bound.
 *   Check that all late variables are either bound or calculated.
 */
final public class CalculationChecker {
    private final Set<Variable> _lateVariables;
    private final Set<Variable> _earlyVariables;

    /**
     * Constructor of class CalculationChecker.
     * <p>
     * Initializes the sets of early and late variables.
     */
    public CalculationChecker() {
        _lateVariables = new HashSet<>();
        _earlyVariables = new HashSet<>();
    }

    /**
     * Resets the state of the calculation checker.
     * <p>
     * Clears the sets of early and late variables.
     */
    public void reset() {
        _lateVariables.clear();
        _earlyVariables.clear();
    }

    /**
     * Adds a variable to the set of early variables.
     *
     * @param var the variable to add
     * @param recorder the state recorder to use
     * @throws Impossible if an impossible state is encountered
     */
    public void addEarlyVariable(final Variable var, StateRecorder recorder) throws Impossible { //NOTICE throws
        if (!_earlyVariables.contains(var)) {
            if (recorder != null) {
                recorder.record(() -> _earlyVariables.remove(var));
            }

            _earlyVariables.add(var);
        }
    }

    /**
     * Inserts a new {@link Calculator} into the unification data structure by
     * unifying it with the given target object.
     * The created calculator uses the given type.
     *
     * @param targetType the type of the target
     * @param target the target object
     * @param source the source object
     * @param recorder the state recorder to use
     * @throws Impossible if the unification fails
     */
    public void addCalculated(
        Class<?> targetType, Object target, Object source, StateRecorder recorder) throws Impossible
    {
        Unify.unify(target, new Calculator(targetType, source, recorder), recorder);
    }

    /**
     * Adds a variable to the set of late variables if it is not already contained.
     *
     * @param var the variable to add
     * @param recorder the state recorder to use
     * @throws Impossible if an impossible state is encountered
     */
    public void addLateVariable(final Variable var, StateRecorder recorder) throws Impossible {
        if (!_lateVariables.contains(var)) {
            if (recorder != null) {
                recorder.record(() -> _lateVariables.remove(var));
            }

            _lateVariables.add(var);
        }
    }

    /**
     * Checks if all late variables are complete.
     *
     * @return if all late variables are complete
     */
    private boolean checkLateVariables() {
        return _lateVariables.stream().allMatch(Variable::isComplete);
    }

    /**
     * Checks if all early variables are bound.
     * (They have to be bound)
     *
     * @return {@code true} if all early variables are bound, {@code false} otherwise
     */
    private boolean checkEarlyVariables() {
        return _earlyVariables.stream().allMatch(Variable::isBound);
    }

    /**
     * The calculation process is consistent if all early variables are bound
     * and if all late variables are complete.
     *
     * @return if the calculation process is consistent
     */
    public boolean isConsistent() {
        return checkLateVariables() && checkEarlyVariables();
    }
}