package de.renew.unify;

import java.util.Vector;

/**
 * This class records state changes and allows them to be restored to a previous checkpoint.
 * <p>
 * Each change is represented by a {@link StateRestorer}, which can restore
 * the state before the change.
 */
public class StateRecorder {
    /** Stores changes to the state that can be reverted. */
    private final Vector<StateRestorer> _savedObjects;

    /**
     * Constructor for the StateRecorder.
     * Initializes an empty structure to record the states.
     */
    public StateRecorder() {
        _savedObjects = new Vector<>();
    }

    /**
     * Records a state change that can later be undone.
     *
     * @param obj the {@code StateRestorer} that encapsulates how to undo a specific change
     */
    public void record(StateRestorer obj) {
        _savedObjects.addElement(obj);
    }

    /**
     * Returns a checkpoint representing the current state of the recorder.
     * This checkpoint can be used later to restore to this exact point.
     *
     * @return the index of the checkpoint, can be used later to restore to this state
     */
    public int checkpoint() {
        return _savedObjects.size();
    }

    /**
     * Restores the system to the state it was in at the given checkpoint.
     *
     * @param checkpoint the index of the checkpoint to restore
     */
    public void restore(int checkpoint) {
        if (checkpoint < _savedObjects.size()) {
            for (int i = _savedObjects.size() - 1; i >= checkpoint; i--) {
                _savedObjects.elementAt(i).restore();
            }
            _savedObjects.setSize(checkpoint);
        }
    }

    /**
     * Restores all recorded changes by backtracking to the initial state.
     */
    public void restore() {
        restore(0);
    }
}