package de.renew.util;

import java.util.Enumeration;

/**
 * Custom class implementing the Enumeration interface. Checks if a provided Object is capable of
 * providing enumerator outputs.
 */
public class ClassCheckEnumeration implements Enumeration<Object> {
    /**
     * An Object-extending object which implements the enumeration interface. This object will be tested for
     * enumeration capabilities.
     */
    private Enumeration<? extends Object> source;
    /**
     * A Class against which will be verified that the Enumerator-returned output is of this class
     */
    private Class<?> clazz;
    /**
     * Stores the output of the Enumerator call
     */
    private Object next = null;
    /**
     * boolean used by the nextElement method to quit the while-loop querrying the enumerator for its outputs
     */
    private boolean more = true;

    /**
     * For a given Object, check if it has enumeration capabilities.
     *
     * @param source the Object on which one can call enumerator methods
     * @param clazz the Class whose type objects returned by the enumeration should be
     */
    public ClassCheckEnumeration(Enumeration<? extends Object> source, Class<?> clazz) {
        this.source = source;
        this.clazz = clazz;
        nextElement();
    }

    @Override
    public boolean hasMoreElements() {
        return more;
    }

    /**
     * Verify that the class is capable of enumeration and produces objects of the type set during construction.
     * Terminates when the enumeration produces no more elements or of a different class than submitted.
     *
     * @return the output of the last performed enumerator call, before any class checking has occurred.
     */
    @Override
    public synchronized Object nextElement() {
        Object result = next;
        more = false;
        while (source.hasMoreElements() && !more) {
            next = source.nextElement();
            more = clazz.isInstance(next);
        }
        return result;
    }
}