package de.renew.util;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * A class providing an implementation of the Iterator interface. Contain a static instance of itself that provides
 * all the services. The object of this may be stored as an iterations-capable object which always indicates it has no
 * next object to iterate on, effectively being an iterator without any iteration output at all.
 */
public class EmptyIterator implements Iterator<Object> {
    /**
     * Single static EmptyIterator instance of itself.
     */
    public final static EmptyIterator INSTANCE = new EmptyIterator();

    /**
     * Constructor for EmptyIterator. Empty.
     */
    public EmptyIterator() {}

    /**
     * hasNext always indicates that EmptyIterator has no capability to return a next iteration object.
     * @return boolan false on all calls
     */
    @Override
    public boolean hasNext() {
        return false;
    }

    /**
     * Implementation of {@link Iterator#next()} which will always throw an Exception since the EmptyIterator class
     * does not support any actual iteration. Since {@link #hasNext()} always returns false, this method should
     * not be called in a properly implemented iterator use.
     *
     * @return no actual return element
     * @throws NoSuchElementException on all calls to this method
     */
    @Override
    public Object next() throws NoSuchElementException {
        throw new NoSuchElementException();
    }

    /**
     * Iplementation of {@link Iterator#remove()} which will always throw an Exception since the EmptyIterator class
     * does not support any actual iteration and as such has no next iteration items to remove from the iteration cue.
     * @throws UnsupportedOperationException on all calls to this method
     */
    @Override
    public void remove() throws UnsupportedOperationException {
        throw new UnsupportedOperationException();
    }
}