package de.renew.appleui;

import java.awt.Desktop;
import java.awt.Taskbar;
import java.awt.Taskbar.Feature;
import java.awt.Toolkit;

import de.renew.plugin.PluginAdapter;
import de.renew.plugin.PluginProperties;
import de.renew.plugin.SoftDependency;


/**
 * This plugin provides basic support for the macOS user interface to the
 * plugin system. It requires the Apple-specific extension package
 * <code>com.apple.eawt</code> for compilation and execution.
 * <p>
 * Included features are:
 * <ul>
 * <li> Apple "quit" events (like Command-Q) are mapped to a clean shutdown
 * of the plugin system. </li>
 * <li>
 * </ul>
 * </p>
 * Created: Thu Jul  8  2004
 * Modified: November 2012
 * Modified: July 2014
 *
 * @author Michael Duvigneau
 * @author Lawrence Cabac
 * @author Konstantin Simon Maria Möllers
 * @version 0.4.1
 */
public class AppleUI extends PluginAdapter {
    private Desktop _desktop = null;
    private AppleUIListener _listener = null;
    private SoftDependency _guiDependency;

    /**
     * Creates an AppleUI with the given PluginProperties.
     *
     * @param props the plugin configuration.
     */
    public AppleUI(PluginProperties props) {
        super(props);
    }

    /**
     * Connects to the eawt application object.
     */
    @Override
    public void init() {
        _desktop = Desktop.getDesktop();
        _listener = new AppleUIListener();
        setAboutDisplayer(null);
        _desktop.setAboutHandler(_listener);
        _desktop.addAppEventListener(_listener);
        _desktop.setQuitHandler(_listener);
        _desktop.setOpenFileHandler(_listener);
        _guiDependency =
            new SoftDependency(this, "de.renew.gui", "de.renew.appleui.GuiDependencyListener");

        //Set icon on the taskbar/dock
        if (Taskbar.isTaskbarSupported()) {
            var taskbar = Taskbar.getTaskbar();
            if (taskbar.isSupported(Feature.ICON_IMAGE)) {
                taskbar.setIconImage(
                    Toolkit.getDefaultToolkit()
                        .getImage(getClass().getResource("/de/renew/appleui/images/Renew.png")));
            }
        }
    }

    /**
     * Disconnects from the eawt application object (as far as
     * possible).
     *
     * @return true
     */
    @Override
    public boolean cleanup() {
        if (_desktop != null) {
            _desktop.removeAppEventListener(_listener);
            _desktop = null;
            _listener = null;
        }

        //NOTICEnull
        if (_guiDependency != null) {
            _guiDependency.discard();
            _guiDependency = null;
        }
        return true;
    }

    /**
     * Configures an <code>AboutDisplayer</code>.
     *
     * @param displayer the about box displayer.
     *                  If <code>null</code>, the menu entry is disabled.
     */
    void setAboutDisplayer(AboutDisplayer displayer) {
        if ((_desktop != null) && (_listener != null)) {
            _listener.setAboutDisplayer(displayer);
        }
    }
}