/*
 * Created on Apr 18, 2003
 */

package CH.ifa.draw.framework;

import java.io.File;
import java.util.HashSet;

import CH.ifa.draw.io.SimpleFileFilter;
import de.renew.ioontology.ExtensionFileFilter;
import de.renew.ioontology.FileFilter;

/**
 * This class holds the SimpleFileFilters for a Drawing.
 * It will provide the default SimpleFileFilters.
 *
 * @author Lawrence Cabac
 */
public class FilterContainer {
    private final ExtensionFileFilter _defaultFileFilter;
    private final HashSet<ExtensionFileFilter> _importfileFilters;
    private final HashSet<ExtensionFileFilter> _exportfileFilters;

    /**
     * Initializes empty sets for import and export file filters, and sets the provided filter
     * as the default.
     * @param ff default file filter
     */
    public FilterContainer(ExtensionFileFilter ff) {
        _importfileFilters = new HashSet<>();
        _exportfileFilters = new HashSet<>();
        _defaultFileFilter = ff;

    }

    /**
     * Initializes empty sets for import and export file filters, and sets the provided filter
     * as the default.
     * @param ff default file filter
     */
    public FilterContainer(FileFilter ff) {
        _importfileFilters = new HashSet<>();
        _exportfileFilters = new HashSet<>();
        _defaultFileFilter = new SimpleFileFilter() {
            @Override
            public boolean accept(File f) {
                return ff.accept(f);
            }
        };
        _defaultFileFilter.setDescription(ff.getDescription());

    }

    /**
     * Registers new Import file filters.
     * @param ff file filter to be added
     * @return boolean, {@code true} if successfully added
     */
    public boolean registerImportFileFilter(SimpleFileFilter ff) {
        return _importfileFilters.add(ff);
    }

    /**
     * Registers new Import file filters.
     * @param ff file filter to be added
     * @return boolean, {@code true} if successfully added
     */
    public boolean registerImportFileFilter(FileFilter ff) {

        SimpleFileFilter simpleFileFilter = new SimpleFileFilter() {
            @Override
            public boolean accept(File f) {
                return ff.accept(f);
            }
        };
        simpleFileFilter.setDescription(ff.getDescription());
        return _importfileFilters.add(simpleFileFilter);

    }

    /**
     * Registers new Export file filters.
     * @param ff file filter to be added
     * @return boolean, {@code true} if successfully added
     */
    public boolean registerExportFileFilter(SimpleFileFilter ff) {
        return _exportfileFilters.add(ff);
    }

    /**
     * Registers new Export file filters.
     * @param ff file filter to be added
     * @return boolean, {@code true} if successfully added
     */
    public boolean registerExportFileFilter(FileFilter ff) {

        SimpleFileFilter simpleFileFilter = new SimpleFileFilter() {
            @Override
            public boolean accept(File f) {
                return ff.accept(f);
            }
        };
        simpleFileFilter.setDescription(ff.getDescription());
        return _exportfileFilters.add(simpleFileFilter);

    }

    /**
     * Deregisters Import File Filter.
     * @param ff file filter to be removed
     * @return boolean, {@code true} if successfully removed.
     */
    public boolean deregisterImportFileFilter(SimpleFileFilter ff) {
        return _importfileFilters.remove(ff);
    }

    /**
     * Deregisters Import File Filter.
     * @param ff file filter to be removed
     * @return boolean, {@code true} if successfully removed.
     */
    public boolean deregisterImportFileFilter(FileFilter ff) {

        SimpleFileFilter simpleFileFilter = new SimpleFileFilter() {
            @Override
            public boolean accept(File f) {
                return ff.accept(f);
            }
        };
        simpleFileFilter.setDescription(ff.getDescription());
        return _importfileFilters.remove(simpleFileFilter);

    }

    /* (non-Javadoc)
     * @see de.renew.diagram.drawing.FilterContainer#getFileFilter()
     */

    /**
     * Gets the import file filters.
     * @return import file filters
     */
    public HashSet<ExtensionFileFilter> getImportFileFilters() {
        return _importfileFilters;
    }

    /* (non-Javadoc)
     * @see de.renew.diagram.drawing.FilterContainer#getFileFilter()
     */

    /**
     * Gets the export file filters.
     * @return export file filters
     */
    public HashSet<ExtensionFileFilter> getExportFileFilters() {
        return _exportfileFilters;
    }

    /* (non-Javadoc)
    * @see de.renew.diagram.drawing.FilterContainer#getDefaultFileFilter()
    */

    /**
     * Gets the default Filter.
     * @return default filter
     */
    public ExtensionFileFilter getDefaultFileFilter() {
        return _defaultFileFilter;
    }
}