/*
 * Created on Apr 13, 2003
 */
package CH.ifa.draw.io;

import java.io.File;
import java.util.Objects;
import javax.swing.filechooser.FileFilter;

import de.renew.ioontology.ExtensionFileFilter;
import de.renew.util.StringUtil;


/**
 * SimpleFileFilter is the class that is the basis for all FileFilters that
 * can be connected to the file types of the Drawings.
 * <p>
 * The SimpleFileFilter is used by the JFileChooser to determine the files that are displayed.
 * Implementations of SimpleFileFilter provide the extension and the description of the
 * denoted filetype. To define FileFilter that handle multiple extensions see
 * CombinationFileFilter.
 * </p>
 *
 * @author Lawrence Cabac
 *
 */
public class SimpleFileFilter extends FileFilter
    implements java.io.FileFilter, ExtensionFileFilter
{
    private String ext;
    private String description;
    private boolean allowDirectory = true;
    private boolean allowHidden = false;

    /**
     * Constructs a SimpleFileFilter with a file extension and a description of the filetype to be filtered.
     *
     * @param extension The file extension
     * @param description description of filetype
     */
    public SimpleFileFilter(String extension, String description) {
        setExtension(extension);
        setDescription(description);
    }

    /**
     * Constructor without parameters.
     */
    protected SimpleFileFilter() {}

    /**
     * Sets whether the SimpleFileFilter will accept hidden files.
     * @param b boolean for allowance
    * */
    public void allowHidden(boolean b) {
        allowHidden = b;
    }

    /**
     * Sets whether the SimpleFileFilter will accept directories as input.
     * @param b boolean for allowance
     * */
    public void allowDirectory(boolean b) {
        allowDirectory = b;
    }

    /**
     * Returns if the SimpleFileFilter will accept hidden files.
     * @return boolean for allowance
     * */
    public boolean isHiddenAllowed() {
        return allowHidden;
    }

    /**
     * Returns if the SimpleFileFilter will accept directories as input.
     * @return boolean for allowance
     * */
    public boolean isDirectoryAllowed() {
        return allowDirectory;
    }

    /**
    * A file is accepted by a SimpleFileFilter when its extension equals the extension
    * of the SimpleFileFilter.
    * @return true for a file that is accepted by the FileFilter.
    * {@link FileFilter#accept(File)}
    */
    @Override
    public boolean accept(File f) {
        if (f != null && (allowHidden || !f.isHidden() && !f.getName().startsWith("."))) {
            if (f.isDirectory()) {
                return allowDirectory;
            }

            String extension = StringUtil.getExtension(f.getName());

            return ext.equals(extension);
        }
        return false;
    }

    /**
     * The description of a file type is a human-readable identifier that is
     * used in the select box of the JFileChooser.
     * @return The Description of the file type.
     * {@link javax.swing.JFileChooser}
     * {@link FileFilter#getDescription()}
     */
    @Override
    public String getDescription() {
        return description;
    }

    /**
     * Returns the file extension of the file type.
     * @return The file extension
     */
    public String getExtension() {
        return ext;
    }

    /**
     * Sets the text for the description of the SimpleFileFilter.
     *
     * @param string The text for the description
     * */
    public void setDescription(String string) {
        description = string;
    }

    /**
     * Sets the text for the extension of the SimpleFileFilter
     *
     * @param string The text for the extension
     * */
    public void setExtension(String string) {
        ext = string;
    }

    /**
     * Two SimpleFileFilter are equal when the extensions of them are equal.
     * @param o The SimpleFileFilter that is compared.
     * @return true if the extensions of the two SimpleFileFilter are equal.
     */
    @Override
    public boolean equals(Object o) {
        return o instanceof SimpleFileFilter
            && Objects.equals(((SimpleFileFilter) o).getExtension(), getExtension());

    }

    @Override
    public int hashCode() {
        return super.hashCode() ^ Objects.hashCode(getExtension());
    }
}