/*
 * @(#)ColorMap.java 5.1
 *
 */

package CH.ifa.draw.util;

import java.awt.Color;


class ColorEntry {
    public String fName;
    public Color fColor;

    ColorEntry(String name, Color color) {
        fColor = color;
        fName = name;
    }
}


/**
 * A map that is filled with some standard colors. The colors
 * can be looked up by name or index.
 *
 */
public class ColorMap {
    /**
     * Reference color for checking transparency.
     */
    public static final Color NONE = new Color(0xFFC79E);

    /**
     * The background color, which is set to white.
     */
    public static final Color BACKGROUND = new Color(0xFEFEFE);
    static ColorEntry[] fMap = {
        new ColorEntry("Black", Color.black), new ColorEntry("Blue", Color.blue),
        new ColorEntry("Green", Color.green), new ColorEntry("Red", Color.red),
        new ColorEntry("Pink", Color.pink), new ColorEntry("Magenta", Color.magenta),
        new ColorEntry("Orange", Color.orange), new ColorEntry("Yellow", Color.yellow),
        new ColorEntry("New Tan", new Color(0xEBC79E)),
        new ColorEntry("Aquamarine", new Color(0x70DB93)),
        new ColorEntry("Sea Green", new Color(0x238E68)),
        new ColorEntry("Dark Gray", Color.darkGray), new ColorEntry("Light Gray", Color.lightGray),
        new ColorEntry("White", Color.white), new ColorEntry("None", NONE),
        new ColorEntry("Background", BACKGROUND) };

    /**
     * Gets the total number of colors saved.
     *
     * @return amount of available colors
     */
    public static int size() {
        return fMap.length;
    }

    /**
     * Gets the color at the specified index.
     *
     * @param index index of the color
     * @return the color object at the specified index
     * @throws ArrayIndexOutOfBoundsException if specified index is greater than length of color map
     *         or negative
     */
    public static Color color(int index) {
        if (index < size() && index >= 0) {
            return fMap[index].fColor;
        }

        throw new ArrayIndexOutOfBoundsException("Color index: " + index);
    }

    /**
     * Returns the color equal to the specified name and returns black if no color was found.
     *
     * @param name the name of the color as String
     * @return the color object equal to the specified name<br>
     *         <code>black</code> if no corresponding color was found
     */
    public static Color color(String name) {
        for (ColorEntry colorEntry : fMap) {
            if (colorEntry.fName.equals(name)) {
                return colorEntry.fColor;
            }
        }

        return Color.black;
    }

    /**
     * Gets the name of the color at the specified index as String.
     *
     * @param index index of the color
     * @return the name of the color at the specified index as String
     * @throws ArrayIndexOutOfBoundsException if specified index is greater than length of color map
     *         or negative
     */
    public static String name(int index) {
        if (index < size() && index >= 0) {
            return fMap[index].fName;
        }

        throw new ArrayIndexOutOfBoundsException("Color index: " + index);
    }

    /**
     * Gets the index of a given color object and returns 0 if the color object wasn't found.
     *
     * @param color the color object to be found in map of all colors
     * @return the index of the color object<br>
     *         0 if color object wasn't found
     */
    public static int colorIndex(Color color) {
        for (int i = 0; i < fMap.length; i++) {
            if (fMap[i].fColor.equals(color)) {
                return i;
            }
        }
        return 0;
    }

    /**
     * Check whether the specified color is equal to the color Object {@link #NONE}
     * held by this class.
     *
     *
     * @param color the color object to be evaluated
     * @return <code>true</code> if color is equal to {@link #NONE}
     */
    public static boolean isTransparent(Color color) {
        return color.equals(NONE);
    }

    /**
     * Check whether the specified color is equal to {@link #BACKGROUND}.
     *
     * @param color the color object to be evaluated
     * @return <code>true</code> if color equal to {@link #BACKGROUND}
     */
    public static boolean isBackground(Color color) {
        return color.equals(BACKGROUND);
    }

    /**
     * Gets a variant of the given color to increase contrast and use as a highlight color.
     * <br>
     * For {@link #BACKGROUND}, black is returned. For black, red is returned. Otherwise, the given
     * color is slightly changed. If the color is already bright, it is changed to a
     * darker version, otherwise it is changed to a brighter version.
     *
     * @param originalColor the original color to darken
     * @return the highlighted version of the given color
     */
    public static Color hilight(Color originalColor) {
        if (isBackground(originalColor)) {
            return Color.black;
        }

        if (originalColor.equals(Color.black)) {
            return Color.red;
        }
        Color hilightColor = originalColor.brighter();
        if (hilightColor.equals(originalColor)) {
            hilightColor = originalColor.darker(); //Color.red.brighter();
        }

        // logger.debug("Color: "+originalColor+"  hilighted: "+hilightColor);
        return hilightColor;
    }
}