package de.renew.draw.storables.impl.drawings;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import de.renew.draw.storables.ontology.Drawing;
import de.renew.ioontology.exporting.ExportFormat;
import de.renew.ioontology.exporting.ExportFormatHolder;
import de.renew.ioontology.exporting.ExportFormatListener;

/**
 * Maintains a list of all known export formats ({@link ExportFormat}),
 * maintains the export menus and offers convenience methods to export to a specific format.
 */
public class DrawingExportFormatHolder implements ExportFormatHolder<Drawing> {
    /**
     * Creates log4j Logger for this class to represent logging information.
     */
    private static final Logger LOGGER = Logger.getLogger(DrawingExportFormatHolder.class);

    private static DrawingExportFormatHolder _instance;

    private final List<ExportFormat<Drawing>> _exportFormats;

    private final List<ExportFormatListener<Drawing>> _exportFormatListeners;


    /**
     * Constructs and initializes an DrawingExportFormatHolder.
     */
    private DrawingExportFormatHolder() {
        _exportFormats = new ArrayList<ExportFormat<Drawing>>();
        _exportFormatListeners = new ArrayList<ExportFormatListener<Drawing>>();
    }

    /**
     * Singleton access method.
     *
     * @return The DrawingExportFormatHolder instance
     */
    public static DrawingExportFormatHolder getInstance() {
        if (_instance == null) {
            _instance = new DrawingExportFormatHolder();
        }
        return _instance;
    }

    /**
     * Adds an ExportFormat for Drawings to Renew.
     * It is added to the known export formats and the export menus.
     *
     * @param exportFormat the ExportFormat which is to be added to Renew
     */
    @Override
    public void addExportFormat(ExportFormat<Drawing> exportFormat) {
        LOGGER.debug(getClass() + ": adding export format " + exportFormat);

        _exportFormats.add(exportFormat);
        for (ExportFormatListener<Drawing> listener : _exportFormatListeners) {
            listener.exportFormatAdded(exportFormat);
        }
    }

    @Override
    public void addExportFormatListener(ExportFormatListener<Drawing> listener) {
        _exportFormatListeners.add(listener);
    }

    @Override
    public List<ExportFormat<Drawing>> getExportFormats() {
        return _exportFormats;
    }

    /**
     * Removes an ExportFormat for Drawings from Renew.
     * It is removed from the known export formats and the export menus.
     *
     * @param exportFormat the ExportFormat which is to be removed from Renew
     */
    @Override
    public void removeExportFormat(ExportFormat<Drawing> exportFormat) {
        LOGGER.debug(getClass() + ": removing export format " + exportFormat);
        _exportFormats.remove(exportFormat);
        for (ExportFormatListener<Drawing> listener : _exportFormatListeners) {
            listener.exportFormatRemoved(exportFormat);
        }
    }

    @Override
    public void removeExportFormatListener(ExportFormatListener<Drawing> listener) {
        _exportFormatListeners.remove(listener);
    }

}