package de.renew.draw.ui.api;

import de.renew.draw.storables.ontology.Drawing;
import de.renew.draw.ui.api.services.UndoRedoService;
import de.renew.draw.ui.impl.services.UndoRedoServiceImpl;

/**
 * API class for managing undo and redo operations.
 */
public final class UndoRedoApi {

    private static UndoRedoService _service;

    // The static initialization that creates the service object
    static {
        _service = new UndoRedoServiceImpl();
    }

    private UndoRedoApi() {
        // As this class only provides static functionality,
        // no instances are allowed to be created.
    }

    /**
     * Takes a snapshot of the current drawing and
     * remembers it until it will be added by {@link #commitUndoSnapshot}.
     * Any previously prepared snapshot will be forgotten.
     */
    public static void prepareUndoSnapshot() {
        _service.prepareUndoSnapshot();
    }

    /**
     * Takes a snapshot of the given drawing and
     * remembers it until it will be added by {@link #commitUndoSnapshot}.
     * Any previously prepared snapshot will be forgotten.
     * @param drawing drawing to be snapshotted
     */
    public static void prepareUndoSnapshot(Drawing drawing) {
        _service.prepareUndoSnapshot(drawing);
    }

    /**
     * Takes the last prepared snapshot and
     * appends it to the undo history of the current drawing.
     * The redo history is cleared.
     */
    public static void commitUndoSnapshot() {
        _service.commitUndoSnapshot();
    }

    /**
     * Takes the last prepared snapshot and
     * appends it to the undo history of the given drawing.
     * The redo history is cleared.
     * @param drawing drawing that has a snapshot to commit
     */
    public static void commitUndoSnapshot(Drawing drawing) {
        _service.commitUndoSnapshot(drawing);
    }

    /**
     * Clears undo and redo history for the current drawing.
     */
    public static void clearUndoRedoHistory() {
        _service.clearUndoRedoHistory();
    }

    /**
     * Clears undo and redo history for the given drawing.
     * @param drawing drawing with undo and redo history
     */
    public static void clearUndoRedoHistory(Drawing drawing) {
        _service.clearUndoRedoHistory(drawing);
    }

    /**
     * Restores the current drawing to the state saved by the last
     * call to {@link #commitUndoSnapshot()}.
     * Additional calls to this method will restore more
     * undo snapshots step by step, until the history is empty.
     */
    public static void restoreUndoSnapshot() {
        _service.restoreUndoSnapshot();
    }

    /**
     * Restores the given drawing to the state saved by the last
     * call to {@link #commitUndoSnapshot(Drawing)}.
     * Additional calls to this method will restore more
     * undo snapshots step by step, until the history is empty.
     * @param drawing drawing whose undo history should be used for restoration
     */
    public static void restoreUndoSnapshot(Drawing drawing) {
        _service.restoreUndoSnapshot(drawing);
    }
}
