package de.renew.draw.ui.api.services;

import de.renew.draw.storables.ontology.Drawing;

/**
 * Service interface that defines methods for the {@link de.renew.draw.ui.api.UndoRedoApi} to call.
 *
 * @author Clara von Bargen
 */
public interface UndoRedoService {

    /**
     * Takes a snapshot of the current drawing and
     * remembers it until it will be added by {@link #commitUndoSnapshot}.
     * Any previously prepared snapshot will be forgotten.
     */
    void prepareUndoSnapshot();

    /**
     * Takes a snapshot of the given drawing and
     * remembers it until it will be added by {@link #commitUndoSnapshot}.
     * Any previously prepared snapshot will be forgotten.
     * @param drawing drawing to be snapshotted
     */
    void prepareUndoSnapshot(Drawing drawing);

    /**
     * Takes the last prepared snapshot and
     * appends it to the undo history of the current drawing.
     * The redo history is cleared.
     */
    void commitUndoSnapshot();

    /**
     * Takes the last prepared snapshot and
     * appends it to the undo history of the given drawing.
     * The redo history is cleared.
     * @param drawing drawing that has a snapshot to commit
     */
    void commitUndoSnapshot(Drawing drawing);

    /**
     * Clears undo and redo history for the current drawing.
     */
    void clearUndoRedoHistory();

    /**
     * Clears undo and redo history for the given drawing.
     * @param drawing drawing with undo and redo history
     */
    void clearUndoRedoHistory(Drawing drawing);

    /**
     * Restores the current drawing to the state saved by the last
     * call to {@link #commitUndoSnapshot()}.
     * Additional calls to this method will restore more
     * undo snapshots step by step, until the history is empty.
     */
    void restoreUndoSnapshot();

    /**
     * Restores the given drawing to the state saved by the last
     * call to {@link #commitUndoSnapshot(Drawing)}.
     * Additional calls to this method will restore more
     * undo snapshots step by step, until the history is empty.
     * @param drawing drawing whose undo history should be used for restoration
     */
    void restoreUndoSnapshot(Drawing drawing);
}
