package de.renew.draw.ui.ontology;

import java.util.Optional;
import java.util.Vector;

import de.renew.draw.storables.ontology.Drawing;
import de.renew.draw.storables.ontology.Figure;

/**
 * An error state holds information regarding an erroneous state.
 * <p>
 * It contains a message and a title, and can hold an offending drawing and its offending figures.
 * Specific information can additionally be registered as any object.
 */
public interface ErrorState {

    /**
     * Gives the message of the error state.
     * @return message
     */
    String getMessage();

    /**
     * Gives the title of the error state.
     * @return title
     */
    String getTitle();

    /**
     * Gives the specific information for the erroneous state if present.
     * @return problem specific information / {@code Optional.empty()} if not present
     */
    Optional<Object> getProblemSpecificInformation();

    /**
     * Gives the erroneous drawing of the error state, if present.
     * @return drawing / {@code Optional.empty()} if not present
     */
    Optional<Drawing> getErroneousDrawing();

    /**
     * Gives a vector of the erroneous figures of the error state.
     * @return vector of erroneous figures
     */
    Vector<Figure> getErroneousFigures();

    /**
     * Gives the specific erroneous figure that was registered, if its present and matches the given type.
     * @param <F> The type for which the erroneous figure class is to be determined.
     * @param figureClass the class object for the type
     * @return erroneous figure / {@code Optional.empty()} if not present or not of given type.
     */
    <F extends Figure> Optional<F> getErroneousFigure(Class<F> figureClass);

    /**
     * Gives the line index that can be used to determine in which line the text is erroneous
     * if a text figure is given as the specific erroneous figure.
     * @return line index
     */
    int getLine();

    /**
     * Gives the column index that can be used to determine at which column the text is erroneous
     * if a text figure is given as the specific erroneous figure.
     * @return column index
     */
    int getColumn();

}
