/*
 * Created on Apr 13, 2003
 */
package de.renew.ioontology;

import java.io.File;
import java.util.Objects;

import de.renew.util.StringUtil;


/**
 * ExtensionFileFilterImpl is the class that is the basis for all FileFilters that
 * can be connected to the file types of the Drawings.
 * <p>
 * The ExtensionFileFilterImpl can be used by the JFileChooser to determine the files that are displayed.
 * To define FileFilter that handle multiple extensions see
 * {@link MultiExtensionFileFilter}.
 * </p>
 *
 * @author Lawrence Cabac
 *
 */
public class ExtensionFileFilterImpl extends javax.swing.filechooser.FileFilter
    implements ExtensionFileFilter
{
    private final String _extension;
    private String _description;

    /**
     * Constructs a ExtensionFileFilter with a file extension and a description of the filetype to be filtered.
     *
     * @param extension The file extension
     * @param description description of filetype
     */
    public ExtensionFileFilterImpl(String extension, String description) {
        _extension = extension;
        _description = description;
    }

    /**
     * Constructs a ExtensionFileFilter with a given file type.
     *
     * @param fileType the file type
     */
    public ExtensionFileFilterImpl(FileType fileType) {
        _extension = fileType.getExtension();
        _description = fileType.getDescription();
    }

    /**
    * A file is accepted by an ExtensionFileFilter when its extension equals the extension
    * of the ExtensionFileFilter.
    * @return true for a file that is accepted by the FileFilter.
    */
    @Override
    public final boolean accept(File f) {
        // do not allow null or hidden files
        if (f == null || f.isHidden() || f.getName().startsWith(".")) {
            return false;
        }

        // allow directories
        if (f.isDirectory()) {
            return true;
        }

        String extension = StringUtil.getExtension(f.getName());
        return _extension.equals(extension);
    }

    /**
     * The description of a file type is a human-readable identifier that can be
     * used in the select box of the {@link javax.swing.JFileChooser}.
     * @return The Description of the file type.
     */
    @Override
    public final String getDescription() {
        return _description;
    }

    @Override
    public final String getExtension() {
        return _extension;
    }

    @Override
    public final void setDescription(String string) {
        _description = string;
    }

    /**
     * Two ExtensionFileFilterImpl are equal when the extensions of them are equal.
     * @param o The ExtensionFileFilterImpl that is compared.
     * @return true if the extensions of the two ExtensionFileFilterImpl are equal.
     */
    @Override
    public final boolean equals(Object o) {
        return o instanceof ExtensionFileFilterImpl
            && Objects.equals(((ExtensionFileFilterImpl) o).getExtension(), getExtension());
    }

    @Override
    public final int hashCode() {
        return Objects.hashCode(getExtension());
    }
}