package de.renew.ioontology.exporting;

import java.io.File;
import java.util.Objects;

import de.renew.ioontology.FileFilter;


/**
 * Abstract implementation of the ExportFormat interface.
 * @param <T> The type of the object that is exported
 */
public abstract class ExportFormatAbstract<T> implements ExportFormat<T> {
    // The name of the format.
    private String _formatName;

    // The FileFilter of the format.
    private FileFilter _fileFilter;

    /**
     * Constructs ExportFormatAbstract with a given name and no  file filter.
     * @param name the given name
     */
    public ExportFormatAbstract(String name) {
        setFormatName(name);
    }

    /**
     * Constructs ExportFormatAbstract with a given name and a given file filter.
     * @param name the given name
     * @param fileFilter the given file filter
     */
    public ExportFormatAbstract(String name, FileFilter fileFilter) {
        setFormatName(name);
        setFileFilter(fileFilter);
    }

    /**
     * Sets _fileFilter to fileFilter.
     * Ensures the given FileFilter is not {@code null}.
     * @param fileFilter the value to be set
     */
    private void setFileFilter(FileFilter fileFilter) {
        _fileFilter = fileFilter;
        Objects.requireNonNull(fileFilter, "Failure in ExportFormatAbstract: fileFilter() == null");
    }

    /**
     * Sets _formatName to formatName.
     * Ensures the given formatName is not {@code null}.
     * @param formatName the value to be set
     */
    private void setFormatName(String formatName) {
        _formatName = formatName;
        Objects
            .requireNonNull(formatName(), "Failure in ExportFormatAbstract: formatName() == null");
    }


    //	---------------------------------------------------------------------
    // Implementation of the ExportFormat interface
    // ---------------------------------------------------------------------

    @Override
    public FileFilter fileFilter() {
        return _fileFilter;
    }

    @Override
    public String formatName() {
        return _formatName;
    }

    @Override
    public File[] exportAll(T[] objects, File[] path) throws Exception {
        File[] result = null;
        if (objects != null && path != null) {
            if (objects.length == path.length) {
                result = new File[path.length];
                for (int pos = 0; pos < objects.length; pos++) {
                    result[pos] = export(objects[pos], path[pos]);
                }
            }
        }
        Objects.requireNonNull(result, "Failure in ExportFormatAbstract : result == null");
        return result;
    }

    @Override
    public ExportFormat<T>[] canExport(File path) {
        Objects.requireNonNull(path, "Failure in ExportFormatAbstract: path == null");
        ExportFormat<T>[] result;

        if (fileFilter().accept(path)) {
            result = new ExportFormat[] { this };
        } else {
            result = new ExportFormat[0];
        }

        return result;
    }

    @Override
    public int getShortCut() {
        return -1;
    }

    @Override
    public int getModifier() {
        return -1;
    }

    @Override
    public boolean forceGivenName() {
        return false;
    }

    @Override
    public String toString() {
        return formatName();
    }
}