package de.renew.ioontology.importing;

import java.io.File;
import java.net.URI;
import java.net.URL;
import java.util.Objects;

import de.renew.ioontology.FileFilter;


/**
 * Abstract implementation of the ImportFormat interface.
 * @param <T> The type of the object that is the result of the import
 */
public abstract class ImportFormatAbstract<T> implements ImportFormat<T> {
    // The name of the format.
    private String _formatName;

    // The FileFilter of the format.
    private FileFilter _fileFilter;

    /**
     * Constructs ImportFormat with a given format name and a given file filter.
     * @param name The given format name
     * @param fileFilter The given file filter
     */
    public ImportFormatAbstract(String name, FileFilter fileFilter) {
        setFormatName(name);
        setFileFilter(fileFilter);
    }

    /**
     * Sets the file filter to fileFilter.
     * Ensures the given fileFilter is not {@code null}.
     * @param fileFilter the value to be set
     */
    private void setFileFilter(FileFilter fileFilter) {
        _fileFilter = fileFilter;
        Objects
            .requireNonNull(fileFilter(), "Failure in ImportFormatAbstract: fileFilter() == null");
    }

    /**
     * Sets the format name to formatName.
     * Ensures the given formatName is not {@code null}.
     * @param formatName the value to be set
     */
    private void setFormatName(String formatName) {
        _formatName = formatName;
        Objects
            .requireNonNull(formatName(), "Failure in ImportFormatAbstract: formatName() == null");
    }

    // ---------------------------------------------------------------------
    // Implementation of the ImportFormat interface
    // ---------------------------------------------------------------------


    @Override
    public FileFilter fileFilter() {
        return _fileFilter;
    }

    @Override
    public String formatName() {
        return _formatName;
    }


    @Override
    public boolean canImport(URL path) {
        return fileFilter().accept(new File(path.getFile()));
    }

    @Override
    public boolean canImport(URI path) {
        return fileFilter().accept(new File(path.getPath()));
    }

    @Override
    public String toString() {
        return formatName();
    }
}