package de.renew.export.io.exportFormats;

import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Rectangle;
import java.io.File;

import org.freehep.graphics2d.VectorGraphics;
import org.freehep.graphicsbase.util.UserProperties;
import org.freehep.graphicsio.PageConstants;
import org.freehep.graphicsio.pdf.PDFGraphics2D;

import de.renew.draw.storables.ontology.Drawing;
import de.renew.draw.storables.ontology.exporting.AbstractDrawingExportFormat;
import de.renew.io.api.FileApi;
import de.renew.ioontology.FileType;
import de.renew.plugin.PluginProperties;

/**
 * @author Benjamin Schleinzer, Michael Haustermann
 */
public class PDFExportFormat extends AbstractDrawingExportFormat {
    public static final org.apache.log4j.Logger LOGGER =
        org.apache.log4j.Logger.getLogger(PDFExportFormat.class);

    //Hotkey property names
    public static final String PDF_EXPORT_HOTKEY_PROP_NAME =
        "de.renew.hotkeymap.export.exportcurrentdrawingpdf";

    private PluginProperties _plugincfg;

    public PDFExportFormat() {
        super("PDF", FileApi.createExtensionFileFilter(FileType.PDF));
    }

    public PDFExportFormat(PluginProperties plugincfg) {
        super("PDF", FileApi.createExtensionFileFilter(FileType.PDF));
        this._plugincfg = plugincfg;
    }



    @Override
    public boolean canExportNto1() {
        return false;
    }

    @Override
    public File export(Drawing[] drawings, File path) throws Exception {
        File result = null;
        assert (result != null) : "Failure in PDFExportFormat: result == null";
        return result;
    }

    @Override
    public File export(Drawing drawing, File file) throws Exception {
        Rectangle bounds = drawing.getBounds();

        return internalExport(drawing, file, bounds);
    }

    public File internalExport(Drawing drawing, File path, Rectangle bounds) throws Exception {
        File result = null;

        int x = bounds.x;
        int y = bounds.y;
        int width = bounds.width;
        int height = bounds.height;

        Dimension d = new Dimension(width, height);
        VectorGraphics graphics = new PDFGraphics2D(path, d);

        String pageSize = ExportPlugin.BOUNDING_BOX_PAGE_SIZE;
        String pageOrientation = PageConstants.PORTRAIT;

        ExportPlugin plugin = ExportPlugin.getCurrent();
        if (plugin != null) {
            pageSize = plugin.getPageSize();
            pageOrientation = plugin.getPageOrientation();
        }

        final UserProperties defaultProperties = new UserProperties();
        if (pageSize.equals(ExportPlugin.BOUNDING_BOX_PAGE_SIZE)) {
            defaultProperties.setProperty(PDFGraphics2D.PAGE_SIZE, PDFGraphics2D.CUSTOM_PAGE_SIZE);
            defaultProperties.setProperty(PDFGraphics2D.CUSTOM_PAGE_SIZE, d);
            defaultProperties.setProperty(PDFGraphics2D.PAGE_MARGINS, new Insets(0, 0, 0, 0));
        } else {
            defaultProperties.setProperty(PDFGraphics2D.PAGE_SIZE, pageSize);
            defaultProperties.setProperty(PDFGraphics2D.ORIENTATION, pageOrientation);
        }

        graphics.setProperties(defaultProperties);

        //Start exporting image to PDF
        graphics.startExport();
        //Move to coordinates 0,0
        graphics.translate(-x, -y);
        //Set clipping to the region that was the original image
        //Otherwise we would export white regions around the image
        graphics.clipRect(x, y, width, height);

        //Paint the picture
        drawing.draw(graphics);

        //End exporting
        graphics.endExport();
        result = path;
        assert (result != null) : "Failure in PDFExportFormat: result == null";
        return result;
    }

    @Override
    public boolean canExportDrawing(Drawing drawing) {
        return true;
    }

    @Override
    public int getModifier() {
        if (_plugincfg != null) {
            return PluginProperties.getUserProperties()
                .getHotkeyProperty(
                    PDF_EXPORT_HOTKEY_PROP_NAME,
                    _plugincfg.getHotkeyProperty(PDF_EXPORT_HOTKEY_PROP_NAME))
                .getModifiers();
        }
        return PluginProperties.getUserProperties().getHotkeyProperty(PDF_EXPORT_HOTKEY_PROP_NAME)
            .getModifiers();
    }


    @Override
    public int getShortCut() {
        if (_plugincfg != null) {
            return PluginProperties.getUserProperties()
                .getHotkeyProperty(
                    PDF_EXPORT_HOTKEY_PROP_NAME,
                    _plugincfg.getHotkeyProperty(PDF_EXPORT_HOTKEY_PROP_NAME))
                .getKeyCode();
        }
        return PluginProperties.getUserProperties().getHotkeyProperty(PDF_EXPORT_HOTKEY_PROP_NAME)
            .getKeyCode();
    }

}
