package de.renew.formalism.fs;

import java.util.List;

import de.renew.simulatorontology.shadow.ShadowInscribable;
import de.renew.simulatorontology.shadow.ShadowNet;


/**
 * Represents a concept in a shadow Petri net for the FS formalism.
 * <p>
 * A ShadowConcept stores its name, namespace, feature structure description,
 * documentation, annotations, and optional stereotype. It serves as the model-level
 * representation of a type that can participate in associations and inheritances
 * within a shadow net.
 */
public class ShadowConcept extends ShadowInscribable {

    /**
     * The namespace of the concept.
     */
    private String _namespace;

    /**
     * The local name of the concept, without the namespace.
     */
    private String _name;

    /**
     * The FS definition of the concept.
     */
    private String _approp;

    /**
     * Documentation for the concept.
     */
    private String _conceptDocumentation;

    /**
     * Documentation for each slot of the concept.
     */
    private List<String> _slotDocumentation;

    /**
     * User-provided annotations for the concept.
     */
    private String _conceptAnnotation;

    /**
     * User-provided annotations for each slot of the concept.
     */
    private List<String> _slotAnnotation;

    /**
     * Optional stereotype of the concept.
     */
    private String _stereotype;

    /**
     * Creates an instance of ShadowConcept.
     * <p>
     * If the name contains "::", the part before is treated as the namespace and
     * the part after as the concept's local name. Otherwise, the net's name is used
     * as the namespace. The constructor also stores the feature structure definition,
     * documentation, annotations, and optional stereotype.
     *
     * @param net the shadow net this concept belongs to
     * @param name the name of the concept, optionally with namespace (e.g., "NS::Concept")
     * @param approp the feature structure definition of this concept
     * @param conceptDocumentation documentation text for the concept
     * @param slotDocumentation documentation for individual slots/features
     * @param conceptAnnotation annotations for the concept
     * @param slotAnnotation annotations for individual slots
     * @param stereotype optional stereotype of the concept
     */
    public ShadowConcept(
        ShadowNet net, String name, String approp, String conceptDocumentation,
        List<String> slotDocumentation, String conceptAnnotation, List<String> slotAnnotation,
        String stereotype)
    {
        super(net);
        int end = name.indexOf("::");
        if (end == -1) {
            _namespace = net.getName();
            this._name = name;
        } else {
            _namespace = name.substring(0, end);
            this._name = name.substring(end + 2);
        }
        this._approp = approp;
        this._conceptDocumentation = conceptDocumentation;
        this._slotDocumentation = slotDocumentation;
        this._conceptAnnotation = conceptAnnotation;
        this._slotAnnotation = slotAnnotation;
        this._stereotype = stereotype;
    }

    /**
     * Returns the local name of this concept (without namespace).
     *
     * @return the local concept name
     */
    public String getName() {
        return _name;
    }

    /**
     * Returns the namespace of this concept.
     *
     * @return the namespace
     */
    public String getNamespace() {
        return _namespace;
    }

    /**
     * Returns the fully qualified name of this concept in the form "namespace::name".
     *
     * @return the fully qualified name
     */
    public String getFullName() {
        return _namespace + "::" + _name;
    }

    /**
     * Sets the feature structure definition for this concept.
     *
     * @param approp the feature structure definition string
     */
    public void setApprop(String approp) {
        this._approp = approp;
    }

    /**
     * Returns the feature structure definition of this concept.
     *
     * @return the feature structure definition
     */
    public String getApprop() {
        return _approp;
    }

    @Override
    public void discard() {
        // is there anything else to do here?
    }

    /**
     * Returns the documentation string associated with this concept.
     *
     * @return the concept documentation
     */
    public String getConceptDocumentation() {
        return _conceptDocumentation;
    }

    /**
     * Returns a list of documentation strings for the slots of this concept.
     *
     * @return the slot documentation list
     */
    public List<String> getSlotDocumentation() {
        return _slotDocumentation;
    }

    /**
     * Returns any annotation associated with this concept.
     *
     * @return the concept annotation
     */
    public String getConceptAnnotation() {
        return _conceptAnnotation;
    }

    /**
     * Returns a list of annotations for the slots of this concept.
     *
     * @return the slot annotation list
     */
    public List<String> getSlotAnnotation() {
        return _slotAnnotation;
    }

    /**
     * Returns the stereotype of this concept.
     *
     * @return the concept stereotype
     */
    public String getStereotype() {
        return _stereotype;
    }
}