package de.uni_hamburg.fs;

import collections.CollectionEnumeration;


/**
 * Represents a conceptual type in a feature structure system.
 * A Concept defines the structure and behavior of feature types,
 * including their namespace, features, and relationships with other concepts.
 * It provides methods to query appropriate features, their types,
 * and to check subsumption relationships between concepts.
 * Each concept belongs to a namespace and can be either extensional or not.
 */
public interface Concept extends java.io.Serializable {

    /**
     * Return the name of this Concept.
     *
     * @return the name of this concept
     */
    public String getName();

    /**
     * Return the name of the namespace this Concept belongs to.
     *
     * @return the namespace of this concept
     */
    public String getNamespace();

    /**
     * Return the full name of this Concept in the form namespace::name.
     *
     * @return the fully qualified name of this concept
     */
    public String getFullName();

    /**
     * Return whether this Concept is extensional.
     *
     * @return true if this concept is extensional, false otherwise
     */
    public boolean isExtensional();

    /**
     * Return whether the feature {@literal <featureName>} is appropriate in this Concept.
     *
     * @param featureName the name of the feature
     * @return true if the feature is appropriate, false otherwise
     */
    public boolean isApprop(Name featureName);

    /**
     * Return the required ParsedType for the given feature.
     *
     * @param featureName the name of the feature
     * @return the parsed type appropriate for the feature
     * @throws NoSuchFeatureException if the feature is not found
     */
    public ParsedType appropParsedType(Name featureName) throws NoSuchFeatureException;

    /**
     * Return the required Type for the Value under the given feature.
     *
     * @param featureName the name of the feature
     * @return the type appropriate for the feature
     * @throws NoSuchFeatureException if the feature is not found
     */
    public Type appropType(Name featureName) throws NoSuchFeatureException;

    /**
     * Return an ordered Enumeration of all appropriate features.
     * A non-restrictive Concept returns only an Enumeration of the
     * FeatureNames for which it defines restricted Types (since it
     * cannot return an Enumeration of all possible FeatureNames).
     *
     * @return enumeration of appropriate feature names
     */
    public CollectionEnumeration appropFeatureNames();

    /**
     * Return whether this Concept is-a {@literal <that>} Concept.
     * In other words, return whether this Concept is more special than {@literal <that>} Concept.
     *
     * @param that the Concept to compare against
     * @return true if this concept is a subtype of the given concept
     */
    public boolean isa(Concept that);

    /**
     * Return whether this Concept is-not-a {@literal <that>} Concept.
     * In other words, return whether this Concept is incompatible with {@literal <that>} Concept.
     * This method should only be called for concepts which do not stand in a subsumption relation.
     *
     * @param that the Concept to compare against
     * @return true if this concept is not a subtype or supertype of the given concept
     */
    public boolean isNotA(Concept that);

    /**
     * Returns an enumeration of all extensional superconcepts of this concept.
     * Extensional superconcepts are concepts that can be directly instantiated
     * and are more general than this concept.
     *
     * @return enumeration of extensional superconcepts
     */


    //  public Concept unify(Concept that);
    public ConceptEnumeration extensionalSuperconcepts();
}