package de.renew.lola2;

import java.awt.Color;
import java.io.File;

import CH.ifa.draw.figures.TextFigure;
import CH.ifa.draw.util.ColorMap;
import de.renew.gui.CPNDrawing;
import de.renew.lola2.analysis.DirectLolaResult;
import de.renew.lola2.analysis.LolaAnalyzer;
import de.renew.lola2.gui.LolaGUIHelper;


/**
 * I am a Lola verification request aka. lola task.
 * If I'm given my text, I can determine my type, write myself to a file
 * and callLola to check myself. Afterwards I can color the figure, that
 * defined me in nice colors.
 *
 * @author hewelt
 *
 */
public class LolaTask {

    /**
     * I can produce console output via this logger
     */
    private static org.apache.log4j.Logger _logger =
        org.apache.log4j.Logger.getLogger(LolaTask.class);

    private String _text;
    private String _formula;

    /**
     * What lola says about me, is put into this variable.
     */
    private DirectLolaResult _result;

    /**
     * The TextFigure that defines/represents me in the drawing.
     */
    private final TextFigure _figure;

    /**
     * The net drawing in which I reside.
     */
    private final CPNDrawing _drawing;

    /**
     * I need to know my figure and the drawing in which it is located.
     * So I can callLola.
     * @param fig the TextFigure that defines this task in the drawing
     * @param drawing the CPNDrawing in which this task is located
     */
    public LolaTask(TextFigure fig, CPNDrawing drawing) {
        _figure = fig;
        _drawing = drawing;
        _text = fig.getText();
        _formula = parseText(_text);
    }

    /**
     * Determine which type of task I am, given a text
     *
     * @param text the text to parse for determining the task type
     * @return The type of this task or "" if none determined
     */
    private String parseText(String text) {
        String formula = null;
        if (text.toLowerCase().startsWith("formula")) {
            // cut off "FORMULA"
            formula = text.substring("formula".length()).trim();
        }
        return formula;
    }

    /**
     * Calls Lola, registers the result and returns it to my caller.
     * I instantiate a LolaAnalyzer to check myself, but I can't now
     * the lib folder (where the lola binaries are located). Fortunatly
     * LolaAnalyzer offers a no argument constructor which asks the
     * LolaPlugin for the lib folder.
     * <b>But this can only work, when a LolaPlugin is running, hence it
     * doesn't work from JUnit.</b>
     * @param analyzer the LolaAnalyzer instance to use for checking this task
     * @return a LolaResult containing the verification results
     */
    public DirectLolaResult check(LolaAnalyzer analyzer) {
        File lolaTmpFile = new LolaFileCreator().writeTemporaryLolaFile(_drawing);
        _result = analyzer.callLola(_formula, lolaTmpFile);
        return _result;
    }


    /**
     * Changes the frame color of the figure this task belongs to
     * according to the LolaResult.
     */
    public void colorFigure() {
        if (_result != null) {
            Color statusColor = LolaGUIHelper.getStatusColor(_result.getStatus());
            _figure.setAttribute("FrameColor", statusColor);
        } else {
            _logger.error("[Lola Task] Task seems to be not yet checked.");
        }
    }

    /**
     * Resets the color of the figure to the default color.
     */
    public void resetColor() {
        _figure.setAttribute("FrameColor", ColorMap.NONE);
    }

    /**
     * Returns a string representation of this task.
     * @return the formula string of this task
     */
    @Override
    public String toString() {
        return _formula;
    }

    /**
     * Gets the TextFigure associated with this task.
     * @return the TextFigure that represents this task in the drawing
     */
    public TextFigure getFigure() {
        return _figure;
    }
}
