package de.renew.propertymanagementgui.controller;

import java.awt.BorderLayout;
import java.io.File;
import java.util.Map;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JOptionPane;

import de.renew.propertymanagement.prop.ConfigurableProperty;
import de.renew.propertymanagement.prop.ConfigurablePropertyManager;
import de.renew.propertymanagement.prop.PropertyLoadFailedException;
import de.renew.propertymanagement.prop.PropertySaveFailedException;
import de.renew.propertymanagementgui.gui.BodyComponent;
import de.renew.propertymanagementgui.reactivecomponents.Component;


/**
 * Controller for the settings window.
 */
public class SettingsController {

    private static final int WINDOW_WIDTH = 800;
    private static final int WINDOW_HEIGHT = 500;

    private final ConfigurablePropertyManager _propertyManager;
    private final Map<String, ConfigurableProperty> _properties;

    /**
     * Creates a new SettingsController
     */
    public SettingsController() {
        _propertyManager = ConfigurablePropertyManager.getInstance();
        _properties = _propertyManager.getConfigurableProperties();
    }

    /**
     * Opens the settings window
     * This method is called when the user clicks on the settings button in the main window
     * Load global properties from file if it exists else load local properties
     */
    public void openSettingsWindow() {
        try {
            _propertyManager.loadProperties(
                new File(System.getProperty("user.home") + File.separator + ".renew.properties")
                    .exists());
        } catch (PropertyLoadFailedException e) {
            JOptionPane.showMessageDialog(null, e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
        }

        Component body = new BodyComponent(_properties, () -> {
            try {
                _propertyManager.saveProperties(true);
                JOptionPane.showMessageDialog(
                    null, "Properties saved successfully in your home directory", "Saved",
                    JOptionPane.INFORMATION_MESSAGE);
            } catch (PropertySaveFailedException e) {
                JOptionPane
                    .showMessageDialog(null, e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
            }
        }, () -> {
            try {
                _propertyManager.saveProperties(false);
                JOptionPane.showMessageDialog(
                    null, "Properties saved successfully at Renew/dist/config/renew.properties",
                    "Saved", JOptionPane.INFORMATION_MESSAGE);
            } catch (PropertySaveFailedException e) {
                JOptionPane
                    .showMessageDialog(null, e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
            }
        });

        renderWindow(body);
    }

    private void renderWindow(Component body) {
        JFrame window = getWindow(body);
        window.setIconImage(
            new ImageIcon("Gui/src/main/resources/de/renew/gui/images/RENEW.gif").getImage());
        window.setVisible(true);
    }

    private JFrame getWindow(Component content) {
        JFrame window = new JFrame("Settings");
        window.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);

        //Sizing & Position
        window.setSize(WINDOW_WIDTH, WINDOW_HEIGHT);
        window.setLocationRelativeTo(null);

        //Layout
        window.setLayout(new BorderLayout());
        window.add(content.get(), BorderLayout.CENTER);

        return window;
    }


}
