/*
 * Created on 16.05.2003
 *
 */
package de.renew.shadowcompiler;


/**
 * This is a {@code ShadowLookup} extension expressing whether the lookup
 * is for sequential simulation only.
 * Use {@link #getSequentialOnly()} to get information on whether the
 * lookup can be used in parallel simulation.
 *
 * @author 6schumac
 */
public class SequentialOnlyExtension implements ShadowLookupExtension {
    // the factory producing this kind of extension
    private static final Factory FACTORY = new Factory();

    // is this lookup sequential?
    private boolean _sequentialOnly = false;

    /**
     * Returns whether any element in the lookup is known to cause problems when
     * used in parallel simulation.
     *
     * @return {@code true} if any element in the lookup is known to cause
     *         problems in parallel simulation, otherwise {@code false}
     */
    public boolean getSequentialOnly() {
        return _sequentialOnly;
    }

    /**
     * Sets the flag to indicate whether the {@link ShadowLookup} can be used
     * in parallel simulation.
     *
     * @param sequential {@code true} if lookup should only be used in sequential
     *                   simulation, otherwise {@code false}
     */
    public void setSequentialOnly(boolean sequential) {
        _sequentialOnly = sequential;
    }

    /**
     * Returns an instance of a {@code SequentialOnlyExtension}, which indicates
     * that this is the {@code ShadowLookupExtension} category for the given lookup.
     *
     * @param lookup the lookup
     * @return an instance of a {@code SequentialOnlyExtension} extension
     */
    public static SequentialOnlyExtension lookup(ShadowLookup lookup) {
        return (SequentialOnlyExtension) lookup.getShadowLookupExtension(FACTORY);
    }

    /**
     * The factory that produces instances of {@code SequentialOnlyExtension}.
     */
    public static class Factory implements ShadowLookupExtensionFactory {

        private static final String SEQUENTIAL_PATH = "de.renew.formalism.sequential";

        @Override
        public String getCategory() {
            return SEQUENTIAL_PATH;
        }

        @Override
        public ShadowLookupExtension createExtension() {
            return new SequentialOnlyExtension();
        }
    }
}