package de.renew.simulatorontology.shadow;

import java.io.Serial;

/**
 * This class is responsible for the inscription of elements of shadow nets.
 */
public class ShadowInscription extends ShadowNetElement {
    /* Serialization support. */
    @Serial
    private static final long serialVersionUID = -8570259322392235092L;
    /** The shadow net element. Needs to be shadow inscribable. */
    private final ShadowInscribable _inscribable;
    /** The inscription of the element. */
    private final String _inscription;
    /** {@code true}, if tracing is desired. */
    private boolean _trace;
    /** {@code true}, if it's special. */
    private boolean _special;

    /**
     * Constructor for a shadow inscription of a shadow net element.
     *
     * @param inscribable net element to have an inscription, needs to be
     *                    shadow inscribable
     * @param inscription inscription for the net element
     */
    public ShadowInscription(ShadowInscribable inscribable, String inscription) {
        super(inscribable.getNet());
        _inscribable = inscribable;
        _inscription = inscription;
        inscribable.add(this);
    }

    /**
     * Switch trace flag on or off.
     *
     * @param trace {@code true}, if tracing is desired
     */
    public void setTrace(boolean trace) {
        if (_trace != trace) {
            _trace = trace;
        }
    }

    /**
     * Returns whether this shadow inscription is being traced.
     *
     * @return {@code true}, if trace flag is set
     */
    public boolean getTrace() {
        return _trace;
    }

    /**
     * Sets whether it is special.
     *
     * @param special {@code true}, if it's special
     */
    public void setSpecial(boolean special) {
        if (_special != special) {
            _special = special;
        }
    }

    /**
     * Returns whether it is special.
     *
     * @return {@code true}, if it's special
     */
    public boolean isSpecial() {
        return _special;
    }

    @Override
    public void discard() {
        getInscribable().remove(this);
        super.discard();
    }

    @Override
    public String toString() {
        if (getInscription() == null) {
            return "ShadowInscription (" + getID() + ") an " + getInscribable();
        } else {
            return "ShadowInscription \"" + getInscription() + "\" an " + getInscribable();
        }
    }

    /**
     * Deserialization method, behaves like default readObject
     * method, additionally re-registers the inscription at its
     * inscribed net element.
     *
     * @param in the object to be read
     * @throws java.io.IOException if the object can't be read
     * @throws ClassNotFoundException if class can't be found while reading the object
     */
    @Serial
    private void readObject(java.io.ObjectInputStream in)
        throws java.io.IOException, ClassNotFoundException
    {
        in.defaultReadObject();
        getInscribable().add(this);
    }

    /**
     * Getter for the inscribable.
     *
     * @return the inscribable
     */
    public ShadowInscribable getInscribable() {
        return _inscribable;
    }

    /**
     * Getter for the inscription.
     *
     * @return the inscription
     */
    public String getInscription() {
        return _inscription;
    }
}