package de.renew.simulatorontology.shadow;


import de.renew.simulatorontology.loading.NetNotFoundException;

/**
 * Shadow net loaders are used to load shadow nets from external sources
 * (like files) to create a new {@link ShadowNetSystem} or insert the contents
 * of the file into an existing one.
 * <p>
 * ShadowNetLoader.java
 * Created: Wed Dec  5  2001
 * @author Michael Duvigneau
 **/
public interface ShadowNetLoader {

    /**
     * Tries to load a shadow net from an external source (for
     * example a file) and creates a new {@code ShadowNetSystem}.
     * <p>
     * This method is called by the {@link DefaultCompiledNetLoader}
     * as a reaction of a call to {@link de.renew.net.Net#forName} if there
     * is no known net for the given name.
     * </p>
     *
     * @param netName the name of the missing net.
     *
     * @return
     *   the new {@code ShadowNetSystem} object, if the loader was
     *   able to find and load the net from an external source.
     *   The net system must include information about the compiler factory
     *   to use.
     *   May not return {@code null} - the loader has to throw
     *   a {@code NetNotFoundException} instead.
     *
     * @throws NetNotFoundException
     *   if the loader could not find a source for a net with the
     *   given name
     **/
    ShadowNetSystem loadShadowNetSystem(String netName) throws NetNotFoundException;

    /**
     * Tries to load a shadow net from an external source (for
     * example a file) and inserts it into the given
     * {@code ShadowNetSystem}.
     * <p>
     * This method is called by the loopback net loader of a {@link ShadowNetSystem}
     * if it is already in the progress of loading a net, that net depends
     * on a net with the given name, and there is no known net for the
     * given name.
     * </p>
     *
     * @param netName the name of the missing net.
     *
     * @param netSystem the net system to put the loaded net into.
     *
     * @return
     *   the new {@code ShadowNet} object, if the loader was
     *   able to find and load the net from an external source.
     *   May not return {@code null} - the loader has to throw
     *   a {@code NetNotFoundException} instead.
     *
     * @throws NetNotFoundException
     *   if the loader could not find a source for a net with the
     *   given name
     **/
    ShadowNet loadShadowNet(String netName, ShadowNetSystem netSystem) throws NetNotFoundException;
}