package de.renew.simulatorontology.shadow;

import java.util.Vector;

/**
 * Represents a syntax error encountered during parsing with optional details
 * such as line, column, detailed messages, and associated objects.
 */
public class SyntaxException extends Exception {
    /** Detailed error messages. */
    private final String[] _detailed;
    /** Line number where the error occurred. */
    private final int _line;
    /** Column number where the error occurred. */
    private final int _column;
    /** Objects related to the error, e.g. graphical elements. */
    private final Vector<Object> _errorObjects;
    /** Problem-specific information object. */
    private Object _object;

    /**
     * Creates a new SyntaxException with the given message, details, position and cause.
     *
     * @param msg the error message
     * @param detailed detailed error descriptions
     * @param line the line number where the error occurred
     * @param column the column number where the error occurred
     * @param cause the cause of the error
     */
    public SyntaxException(String msg, String[] detailed, int line, int column, Throwable cause) {
        super(msg, cause);
        _detailed = detailed;
        _line = line;
        _column = column;
        _errorObjects = new Vector<>();
    }

    /**
     * Creates a new SyntaxException with the given message, details and position.
     *
     * @param msg the error message
     * @param detailed detailed error descriptions
     * @param line the line number
     * @param column the column number
     */
    public SyntaxException(String msg, String[] detailed, int line, int column) {
        this(msg, detailed, line, column, null);
    }

    /**
     * Creates a new SyntaxException with the given message and details.
     *
     * @param msg the error message
     * @param detailed detailed error descriptions
     */
    public SyntaxException(String msg, String[] detailed) {
        this(msg, detailed, 0, 0, null);
    }

    /**
     * Creates a new SyntaxException with the given message and cause.
     *
     * @param msg the error message
     * @param cause the cause of the error
     */
    public SyntaxException(String msg, Throwable cause) {
        this(msg, null, 0, 0, cause);
    }

    /**
     * Creates a new SyntaxException with the given message.
     *
     * @param msg the error message
     */
    public SyntaxException(String msg) {
        this(msg, null, 0, 0, null);
    }

    /**
     * Adds an object related to the error.
     *
     * @param graphicObject the related object to associate
     * @return this exception instance
     */
    public SyntaxException addObject(Object graphicObject) {
        getErrorObjects().addElement(graphicObject);
        return this;
    }

    /**
     * Creates a new SyntaxException with the given message, details, position,
     * cause and problem-specific data.
     *
     * @param msg the error message
     * @param detailed detailed descriptions
     * @param beginLine the starting line
     * @param beginColumn the starting column
     * @param cause the cause
     * @param object problem-specific data
     */
    public SyntaxException(
        String msg, String[] detailed, int beginLine, int beginColumn, Throwable cause,
        Object object)
    {
        this(msg, detailed, beginLine, beginColumn, cause);
        _object = object;
    }

    /**
     * Returns the problem-specific information object, or {@code null} if there
     * is none.
     *
     * @return the problem-specific information object
     */
    public Object getProblemSpecificInformation() {
        return _object;
    }

    /**
     * Getter for the detailed error messages.
     *
     * @return the detailed messages
     */
    public String[] getDetailed() {
        return _detailed;
    }

    /**
     * Getter for the line number.
     *
     * @return the line number
     */
    public int getLine() {
        return _line;
    }

    /**
     * Getter for the column number.
     *
     * @return the column number
     */
    public int getColumn() {
        return _column;
    }

    /**
     * Getter for the error objects.
     *
     * @return the error objects
     */
    public Vector<Object> getErrorObjects() {
        return _errorObjects;
    }
}