package de.renew.splashscreen.api;

import java.util.HashMap;
import java.util.Map;

import de.renew.plugin.ServiceLookupInfrastructure;
import de.renew.plugin.load.ISplashscreenPlugin;
import de.renew.splashscreen.proc.Cleanup;
import de.renew.splashscreen.proc.CloseSplashscreen;
import de.renew.splashscreen.proc.PropertyChange;
import de.renew.splashscreen.proc.SequentialProcess;
import de.renew.splashscreen.proc.ShowSplashscreen;

/**
 * This class acts as an interface to the different processes of Splashscreen.
 *
 * @author Niklas Levens
 * @author Tom Dasenbrook
 * @date Dec 02, 2025
 */
public class ProcessInterface {

    /**
     * Stores all available processes.
     * A process can be accessed by its key.
     */
    private static final Map<String, SequentialProcess> _processLookupTable = new HashMap<>();

    /*
     * Implementations of the processes.
     * These have to be initialized manually, since Splashscreen has to offer its functionality on startup and ServiceLookupArchitecture can't find anything in uninitialized modules.
     */
    private static final SequentialProcess _showSplashscreen = new ShowSplashscreen();
    private static final SequentialProcess _cleanup = new Cleanup();
    private static final SequentialProcess _propertyChange = new PropertyChange();
    private static final SequentialProcess _closeSplashscreen = new CloseSplashscreen();

    private ProcessInterface() {}

    /**
     * Uses the {@link ServiceLookupInfrastructure} to find all implementations of {@link SequentialProcess}.
     * The found implementations are then stored in a private map.
     */
    private static void updateLookupTable() {
        if (_processLookupTable.isEmpty()) {
            _processLookupTable.put(_showSplashscreen.getKey(), _showSplashscreen);
            _processLookupTable.put(_cleanup.getKey(), _cleanup);
            _processLookupTable.put(_propertyChange.getKey(), _propertyChange);
            _processLookupTable.put(_closeSplashscreen.getKey(), _closeSplashscreen);
        }
        for (SequentialProcess p : ServiceLookupInfrastructure.getInstance()
            .getAllServiceProvider(SequentialProcess.class)) {
            _processLookupTable.put(p.getKey(), p);
        }
    }

    /**
     * Execute a process defined by a key with arguments.
     * Will dynamically look for processes which implement the {@link SequentialProcess} interface.
     *
     * @param processKey    The key for the process
     * @param args          Arguments for the process
     * @throws Exception    If no process is found
     */
    public static Object executeProcess(
        String processKey, Object[] args, ISplashscreenPlugin trigger) throws Exception
    {
        if (!_processLookupTable.containsKey(processKey)) {
            //Update registry and check again before throwing exception
            updateLookupTable();
            if (!_processLookupTable.containsKey(processKey)) {
                throw new Exception("Process " + processKey + " not found");
            }
        }
        return _processLookupTable.get(processKey).execute(args, trigger);
    }
}
